/********************************************************************************************
  Copyright (C) 2008 by Mathias Soeken (msoeken@informatik.uni-bremen.de)
            (C) 2005-2006 by Holger Danielsson (holger.danielsson@t-online.de)
            (C) 2012 by Michel Ludwig (michel.ludwig@kdemail.net)
 ********************************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#include "tabularcell.h"

#include "tabularheaderitem.h"
#include "tabularproperties.h"

namespace KileDialog {

//BEGIN TabularCell
TabularCell::TabularCell()
    : QTableWidgetItem(),
      m_Border(None)
{
}

TabularCell::TabularCell(const QString &text)
    : QTableWidgetItem(text),
      m_Border(None)
{
}

void TabularCell::setBorder(int border)
{
    m_Border = border;
    tableWidget()->update();
}

int TabularCell::border() const
{
    return m_Border;
}

QString TabularCell::toLaTeX( TabularProperties &properties ) const
{
    QString prefix;
    QString suffix;

    int alignment = textAlignment() & ~Qt::AlignVCenter;
    TabularHeaderItem *headerItem = static_cast<TabularHeaderItem*>(tableWidget()->horizontalHeaderItem(column()));

    QString colorCommand;
    if(background().style() != Qt::NoBrush
            && !properties.rowColor(row()).isValid()) {
        colorCommand = QStringLiteral(">{\\columncolor{") + properties.colorName(background().color()) + QStringLiteral("}}");
    }

    QString leftBorder, rightBorder;
    // First col border always needs to be set
    if(column() == 0) {
        if(border() & TabularCell::Left) {
            leftBorder = QLatin1Char('|');
        }
    }
    // Does the cell have a right border?
    if(border() & TabularCell::Right) {
        rightBorder = QLatin1Char('|');
    }

    bool adjustBorder = false;
    // If 1st col has no left border, but the cell should have one, set it manually
    if(column() == 0 &&  !properties.hasLeftBorder() &&
            (border() & TabularCell::Left)) {
        adjustBorder = true;
    }
    // Do we have to set the right border manually?
    if(!properties.hasBorderBesideColumn(column()) &&
            (border() & TabularCell::Right)) {
        adjustBorder = true;
    }

    int columnSpan = tableWidget()->columnSpan(row(), column());

    if(headerItem->alignment() != alignment || !colorCommand.isEmpty() ||
            adjustBorder || columnSpan > 1 ) {

        switch(alignment) {
        case Qt::AlignHCenter:
            properties.setUseMultiColumn();
            prefix += QStringLiteral("\\mc{") + QString::number(columnSpan) + QStringLiteral("}{") +
                      leftBorder + colorCommand + QLatin1Char('c') + rightBorder + QStringLiteral("}{");
            suffix = QLatin1Char('}') + suffix;
            break;

        case Qt::AlignRight:
            properties.setUseMultiColumn();
            prefix += QStringLiteral("\\mc{") + QString::number(columnSpan) + QStringLiteral("}{") +
                      leftBorder + colorCommand + QLatin1Char('r') + rightBorder + QStringLiteral("}{");
            suffix = QLatin1Char('}') + suffix;
            break;
        default: // This handles Qt::AlignLeft,
            // alignP, alignM, alignB and alignX (they get thrown away here)
            properties.setUseMultiColumn();
            prefix += QStringLiteral("\\mc{") + QString::number(columnSpan) + QStringLiteral("}{") +
                      leftBorder + colorCommand + QLatin1Char('l') + rightBorder + QStringLiteral("}{");
            suffix = QLatin1Char('}') + suffix;
            break;
        };
    }

    /* format */
    if (font().bold()) {
        prefix += QStringLiteral("\\textbf{");
        suffix = QLatin1Char('}') + suffix;
    }
    if (font().italic()) {
        prefix += QStringLiteral("\\textit{");
        suffix = QLatin1Char('}') + suffix;
    }

    /* prefix */
    if (font().underline()) {
        prefix += QStringLiteral("\\underline{");
        suffix = QLatin1Char('}') + suffix;
    }

    /* foreground color */
    if(foreground().style() != Qt::NoBrush) {
        prefix += QStringLiteral("\\textcolor{") + properties.colorName(foreground().color()) + QStringLiteral("}{");
        suffix = QLatin1Char('}') + suffix;
    }

    /* content */
    QString content = QStringLiteral("");
    QString incontent = text().trimmed();
    if(incontent.isEmpty()) {
        incontent = properties.bullet();
    }
    content += prefix + incontent + suffix;
    return content;
}
//END

}
