/*
    This file is part of Akonadi Contact.

    SPDX-FileCopyrightText: 2010 Tobias Koenig <tokoe@kde.org>

    SPDX-License-Identifier: LGPL-2.0-or-later
*/

#pragma once

#include "akonadi-contact-core_export.h"

#include <QVariantMap>

#include <memory>

namespace KContacts
{
class Addressee;
}

namespace Akonadi
{
class Item;
class AbstractContactFormatterPrivate;

/*!
 * \class Akonadi::AbstractContactFormatter
 * \brief The interface for all contact formatters.
 * \inmodule AkonadiContactCore
 * \inheaderfile Akonadi/AbstractContactFormatter
 *
 * This is the interface that can be used to format an Akonadi
 * item with a contact payload or a contact itself as HTML.
 *
 * \sa StandardContactFormatter
 * \author Tobias Koenig <tokoe@kde.org>
 * \since 4.5
 */
class AKONADI_CONTACT_CORE_EXPORT AbstractContactFormatter
{
public:
    /*!
     * Describes the form of the HTML that is created.
     */
    enum HtmlForm {
        SelfcontainedForm, ///< Creates a complete HTML document
        EmbeddableForm, ///< Creates a div HTML element that can be embedded.
        UserForm = SelfcontainedForm + 42 ///< Point for extension
    };

    /*!
     * Creates a new abstract contact formatter.
     */
    AbstractContactFormatter();

    /*!
     * Destroys the abstract contact formatter.
     */
    virtual ~AbstractContactFormatter();

    /*!
     * Sets the \a contact that will be formatted.
     * \a contact contact to be formatted
     */
    void setContact(const KContacts::Addressee &contact);

    /*!
     * Returns the contact that will be formatted.
     */
    [[nodiscard]] KContacts::Addressee contact() const;

    /*!
     * Sets the \a item who's payload will be formatted.
     *
     * \note The payload must be a valid KContacts::Addressee object.
     * \a item item, who's payload will be formatted.
     */
    void setItem(const Akonadi::Item &item);

    /*!
     * Returns the item who's payload will be formatted.
     */
    [[nodiscard]] Akonadi::Item item() const;

    /*!
     * Sets the custom field \a descriptions that will be used.
     *
     * The description list contains a QVariantMap for each custom field
     * with the following keys:
     *   - key   (string) The identifier of the field
     *   - title (string) The i18n'ed title of the field
     *   - type  (string) The type description of the field
     *     Possible values for type description are
     *       - text
     *       - numeric
     *       - boolean
     *       - date
     *       - time
     *       - datetime
     *
     * \a descriptions list with field descriptions
     */
    void setCustomFieldDescriptions(const QList<QVariantMap> &descriptions);

    /*!
     * Returns the custom field descriptions that will be used.
     */
    [[nodiscard]] QList<QVariantMap> customFieldDescriptions() const;

    /*!
     * This method must be reimplemented to return the contact formatted as HTML
     * according to the requested \a form.
     * \a form how to render the contact into HTML
     */
    virtual QString toHtml(HtmlForm form = SelfcontainedForm) const = 0;

private:
    Q_DISABLE_COPY(AbstractContactFormatter)

    std::unique_ptr<AbstractContactFormatterPrivate> const d;
};
}
