/*
    This file is part of Akonadi Contact.

    SPDX-FileCopyrightText: 2009 Tobias Koenig <tokoe@kde.org>

    SPDX-License-Identifier: LGPL-2.0-or-later
*/

#pragma once

#include "akonadi-contact-widgets_export.h"

#include <Akonadi/ItemMonitor>

#include <QWidget>

#include <memory>

namespace KContacts
{
class Address;
class Addressee;
}

namespace Akonadi
{
class AbstractContactFormatter;
class ContactViewerPrivate;

/*!
 * \class Akonadi::ContactViewer
 * \brief A viewer component for contacts in Akonadi.
 * \inmodule AkonadiContactWidgets
 * \inheaderfile Akonadi/ContactViewer
 *
 * This widgets provides a way to show a contact from the
 * Akonadi storage or a raw contact object.
 *
 * Examples:
 *
 * \code
 *
 * using namespace Akonadi;
 *
 * const Item contact = ...
 *
 * ContactViewer *viewer = new ContactViewer( this );
 * viewer->setContact( contact );
 *
 * \endcode

 * \code
 *
 * using namespace Akonadi;
 *
 * const KContacts::Addressee contact = ...
 *
 * ContactViewer *viewer = new ContactViewer( this );
 * viewer->setContact( contact );
 *
 * \endcode
 *
 * \author Tobias Koenig <tokoe@kde.org>
 * \since 4.4
 */
class AKONADI_CONTACT_WIDGETS_EXPORT ContactViewer : public QWidget, public Akonadi::ItemMonitor
{
    Q_OBJECT

public:
    /*!
     * Creates a new contact viewer.
     *
     * \a parent The parent widget.
     */
    explicit ContactViewer(QWidget *parent = nullptr);

    /*!
     * Destroys the contact viewer.
     */
    ~ContactViewer() override;

    /*!
     * Returns the contact that is currently displayed.
     *
     * \note The returned contact is only valid if it was
     *       set with setContact() before.
     */
    [[nodiscard]] Akonadi::Item contact() const;

    /*!
     * Returns the raw contact that is currently displayed.
     *
     * \since 4.5
     */
    [[nodiscard]] KContacts::Addressee rawContact() const;

    /*!
     * Sets the contact \a formatter that should be used for formatting the
     * contact. If formatter is \\ 0, the standard formatter will be used.
     * \a formatter the contact formatter to set
     * \note The contact viewer won't take ownership of the formatter.
     *
     * \since 4.6
     */
    void setContactFormatter(Akonadi::AbstractContactFormatter *formatter);
    /*!
     * \since 5.1
     */
    void updateView();

    /*!
     * \since 5.2
     */
    void setShowQRCode(bool b);
    [[nodiscard]] bool showQRCode() const;
public Q_SLOTS:
    /*!
     * Sets the \a contact that shall be displayed in the viewer.
     */
    void setContact(const Akonadi::Item &contact);

    /*!
     * Sets the raw \a contact object that shall be displayed in the viewer.
     * \a contact the contact object to set
     * \since 4.5
     */
    void setRawContact(const KContacts::Addressee &contact);

Q_SIGNALS:
    /*!
     * This signal is emitted whenever the user has clicked on
     * a url (e.g. homepage or blog url) in the viewer.
     *
     * \a url The url that has been clicked.
     */
    void urlClicked(const QUrl &url);

    /*!
     * This signal is emitted whenever the user has clicked on an
     * address in the viewer.
     *
     * \a address The corresponding address.
     */
    void addressClicked(const KContacts::Address &address);

private:
    /*!
     * This method is called whenever the displayed contact has been changed.
     */
    void itemChanged(const Akonadi::Item &contact) override;

    /*!
     * This method is called whenever the displayed contact has been
     * removed from Akonadi.
     */
    void itemRemoved() override;

private:
    std::unique_ptr<ContactViewerPrivate> const d;

    Q_PRIVATE_SLOT(d, void slotUrlClicked(const QUrl &))
    Q_PRIVATE_SLOT(d, void slotParentCollectionFetched(KJob *))
};
}
