﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#include <aws/core/Globals.h>
#include <aws/core/utils/EnumParseOverflowContainer.h>
#include <aws/core/utils/HashingUtils.h>
#include <aws/devops-guru/model/OptInStatus.h>

using namespace Aws::Utils;

namespace Aws {
namespace DevOpsGuru {
namespace Model {
namespace OptInStatusMapper {

static const int ENABLED_HASH = HashingUtils::HashString("ENABLED");
static const int DISABLED_HASH = HashingUtils::HashString("DISABLED");

OptInStatus GetOptInStatusForName(const Aws::String& name) {
  int hashCode = HashingUtils::HashString(name.c_str());
  if (hashCode == ENABLED_HASH) {
    return OptInStatus::ENABLED;
  } else if (hashCode == DISABLED_HASH) {
    return OptInStatus::DISABLED;
  }
  EnumParseOverflowContainer* overflowContainer = Aws::GetEnumOverflowContainer();
  if (overflowContainer) {
    overflowContainer->StoreOverflow(hashCode, name);
    return static_cast<OptInStatus>(hashCode);
  }

  return OptInStatus::NOT_SET;
}

Aws::String GetNameForOptInStatus(OptInStatus enumValue) {
  switch (enumValue) {
    case OptInStatus::NOT_SET:
      return {};
    case OptInStatus::ENABLED:
      return "ENABLED";
    case OptInStatus::DISABLED:
      return "DISABLED";
    default:
      EnumParseOverflowContainer* overflowContainer = Aws::GetEnumOverflowContainer();
      if (overflowContainer) {
        return overflowContainer->RetrieveOverflow(static_cast<int>(enumValue));
      }

      return {};
  }
}

}  // namespace OptInStatusMapper
}  // namespace Model
}  // namespace DevOpsGuru
}  // namespace Aws
