﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#include <aws/core/utils/json/JsonSerializer.h>
#include <aws/glue/model/Filter.h>

#include <utility>

using namespace Aws::Utils::Json;
using namespace Aws::Utils;

namespace Aws {
namespace Glue {
namespace Model {

Filter::Filter(JsonView jsonValue) { *this = jsonValue; }

Filter& Filter::operator=(JsonView jsonValue) {
  if (jsonValue.ValueExists("Name")) {
    m_name = jsonValue.GetString("Name");
    m_nameHasBeenSet = true;
  }
  if (jsonValue.ValueExists("Inputs")) {
    Aws::Utils::Array<JsonView> inputsJsonList = jsonValue.GetArray("Inputs");
    for (unsigned inputsIndex = 0; inputsIndex < inputsJsonList.GetLength(); ++inputsIndex) {
      m_inputs.push_back(inputsJsonList[inputsIndex].AsString());
    }
    m_inputsHasBeenSet = true;
  }
  if (jsonValue.ValueExists("LogicalOperator")) {
    m_logicalOperator = FilterLogicalOperatorMapper::GetFilterLogicalOperatorForName(jsonValue.GetString("LogicalOperator"));
    m_logicalOperatorHasBeenSet = true;
  }
  if (jsonValue.ValueExists("Filters")) {
    Aws::Utils::Array<JsonView> filtersJsonList = jsonValue.GetArray("Filters");
    for (unsigned filtersIndex = 0; filtersIndex < filtersJsonList.GetLength(); ++filtersIndex) {
      m_filters.push_back(filtersJsonList[filtersIndex].AsObject());
    }
    m_filtersHasBeenSet = true;
  }
  return *this;
}

JsonValue Filter::Jsonize() const {
  JsonValue payload;

  if (m_nameHasBeenSet) {
    payload.WithString("Name", m_name);
  }

  if (m_inputsHasBeenSet) {
    Aws::Utils::Array<JsonValue> inputsJsonList(m_inputs.size());
    for (unsigned inputsIndex = 0; inputsIndex < inputsJsonList.GetLength(); ++inputsIndex) {
      inputsJsonList[inputsIndex].AsString(m_inputs[inputsIndex]);
    }
    payload.WithArray("Inputs", std::move(inputsJsonList));
  }

  if (m_logicalOperatorHasBeenSet) {
    payload.WithString("LogicalOperator", FilterLogicalOperatorMapper::GetNameForFilterLogicalOperator(m_logicalOperator));
  }

  if (m_filtersHasBeenSet) {
    Aws::Utils::Array<JsonValue> filtersJsonList(m_filters.size());
    for (unsigned filtersIndex = 0; filtersIndex < filtersJsonList.GetLength(); ++filtersIndex) {
      filtersJsonList[filtersIndex].AsObject(m_filters[filtersIndex].Jsonize());
    }
    payload.WithArray("Filters", std::move(filtersJsonList));
  }

  return payload;
}

}  // namespace Model
}  // namespace Glue
}  // namespace Aws
