﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#include <aws/core/Globals.h>
#include <aws/core/utils/EnumParseOverflowContainer.h>
#include <aws/core/utils/HashingUtils.h>
#include <aws/glue/model/RegistryStatus.h>

using namespace Aws::Utils;

namespace Aws {
namespace Glue {
namespace Model {
namespace RegistryStatusMapper {

static const int AVAILABLE_HASH = HashingUtils::HashString("AVAILABLE");
static const int DELETING_HASH = HashingUtils::HashString("DELETING");

RegistryStatus GetRegistryStatusForName(const Aws::String& name) {
  int hashCode = HashingUtils::HashString(name.c_str());
  if (hashCode == AVAILABLE_HASH) {
    return RegistryStatus::AVAILABLE;
  } else if (hashCode == DELETING_HASH) {
    return RegistryStatus::DELETING;
  }
  EnumParseOverflowContainer* overflowContainer = Aws::GetEnumOverflowContainer();
  if (overflowContainer) {
    overflowContainer->StoreOverflow(hashCode, name);
    return static_cast<RegistryStatus>(hashCode);
  }

  return RegistryStatus::NOT_SET;
}

Aws::String GetNameForRegistryStatus(RegistryStatus enumValue) {
  switch (enumValue) {
    case RegistryStatus::NOT_SET:
      return {};
    case RegistryStatus::AVAILABLE:
      return "AVAILABLE";
    case RegistryStatus::DELETING:
      return "DELETING";
    default:
      EnumParseOverflowContainer* overflowContainer = Aws::GetEnumOverflowContainer();
      if (overflowContainer) {
        return overflowContainer->RetrieveOverflow(static_cast<int>(enumValue));
      }

      return {};
  }
}

}  // namespace RegistryStatusMapper
}  // namespace Model
}  // namespace Glue
}  // namespace Aws
