﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#include <aws/core/Globals.h>
#include <aws/core/utils/EnumParseOverflowContainer.h>
#include <aws/core/utils/HashingUtils.h>
#include <aws/resource-explorer-2/model/OperationStatus.h>

using namespace Aws::Utils;

namespace Aws {
namespace ResourceExplorer2 {
namespace Model {
namespace OperationStatusMapper {

static const int SUCCEEDED_HASH = HashingUtils::HashString("SUCCEEDED");
static const int FAILED_HASH = HashingUtils::HashString("FAILED");
static const int IN_PROGRESS_HASH = HashingUtils::HashString("IN_PROGRESS");
static const int SKIPPED_HASH = HashingUtils::HashString("SKIPPED");

OperationStatus GetOperationStatusForName(const Aws::String& name) {
  int hashCode = HashingUtils::HashString(name.c_str());
  if (hashCode == SUCCEEDED_HASH) {
    return OperationStatus::SUCCEEDED;
  } else if (hashCode == FAILED_HASH) {
    return OperationStatus::FAILED;
  } else if (hashCode == IN_PROGRESS_HASH) {
    return OperationStatus::IN_PROGRESS;
  } else if (hashCode == SKIPPED_HASH) {
    return OperationStatus::SKIPPED;
  }
  EnumParseOverflowContainer* overflowContainer = Aws::GetEnumOverflowContainer();
  if (overflowContainer) {
    overflowContainer->StoreOverflow(hashCode, name);
    return static_cast<OperationStatus>(hashCode);
  }

  return OperationStatus::NOT_SET;
}

Aws::String GetNameForOperationStatus(OperationStatus enumValue) {
  switch (enumValue) {
    case OperationStatus::NOT_SET:
      return {};
    case OperationStatus::SUCCEEDED:
      return "SUCCEEDED";
    case OperationStatus::FAILED:
      return "FAILED";
    case OperationStatus::IN_PROGRESS:
      return "IN_PROGRESS";
    case OperationStatus::SKIPPED:
      return "SKIPPED";
    default:
      EnumParseOverflowContainer* overflowContainer = Aws::GetEnumOverflowContainer();
      if (overflowContainer) {
        return overflowContainer->RetrieveOverflow(static_cast<int>(enumValue));
      }

      return {};
  }
}

}  // namespace OperationStatusMapper
}  // namespace Model
}  // namespace ResourceExplorer2
}  // namespace Aws
