﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#include <aws/core/Globals.h>
#include <aws/core/utils/EnumParseOverflowContainer.h>
#include <aws/core/utils/HashingUtils.h>
#include <aws/route53/model/ChangeStatus.h>

using namespace Aws::Utils;

namespace Aws {
namespace Route53 {
namespace Model {
namespace ChangeStatusMapper {

static const int PENDING_HASH = HashingUtils::HashString("PENDING");
static const int INSYNC_HASH = HashingUtils::HashString("INSYNC");

ChangeStatus GetChangeStatusForName(const Aws::String& name) {
  int hashCode = HashingUtils::HashString(name.c_str());
  if (hashCode == PENDING_HASH) {
    return ChangeStatus::PENDING;
  } else if (hashCode == INSYNC_HASH) {
    return ChangeStatus::INSYNC;
  }
  EnumParseOverflowContainer* overflowContainer = Aws::GetEnumOverflowContainer();
  if (overflowContainer) {
    overflowContainer->StoreOverflow(hashCode, name);
    return static_cast<ChangeStatus>(hashCode);
  }

  return ChangeStatus::NOT_SET;
}

Aws::String GetNameForChangeStatus(ChangeStatus enumValue) {
  switch (enumValue) {
    case ChangeStatus::NOT_SET:
      return {};
    case ChangeStatus::PENDING:
      return "PENDING";
    case ChangeStatus::INSYNC:
      return "INSYNC";
    default:
      EnumParseOverflowContainer* overflowContainer = Aws::GetEnumOverflowContainer();
      if (overflowContainer) {
        return overflowContainer->RetrieveOverflow(static_cast<int>(enumValue));
      }

      return {};
  }
}

}  // namespace ChangeStatusMapper
}  // namespace Model
}  // namespace Route53
}  // namespace Aws
