﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#include <aws/core/Globals.h>
#include <aws/core/utils/EnumParseOverflowContainer.h>
#include <aws/core/utils/HashingUtils.h>
#include <aws/scheduler/model/ScheduleState.h>

using namespace Aws::Utils;

namespace Aws {
namespace Scheduler {
namespace Model {
namespace ScheduleStateMapper {

static const int ENABLED_HASH = HashingUtils::HashString("ENABLED");
static const int DISABLED_HASH = HashingUtils::HashString("DISABLED");

ScheduleState GetScheduleStateForName(const Aws::String& name) {
  int hashCode = HashingUtils::HashString(name.c_str());
  if (hashCode == ENABLED_HASH) {
    return ScheduleState::ENABLED;
  } else if (hashCode == DISABLED_HASH) {
    return ScheduleState::DISABLED;
  }
  EnumParseOverflowContainer* overflowContainer = Aws::GetEnumOverflowContainer();
  if (overflowContainer) {
    overflowContainer->StoreOverflow(hashCode, name);
    return static_cast<ScheduleState>(hashCode);
  }

  return ScheduleState::NOT_SET;
}

Aws::String GetNameForScheduleState(ScheduleState enumValue) {
  switch (enumValue) {
    case ScheduleState::NOT_SET:
      return {};
    case ScheduleState::ENABLED:
      return "ENABLED";
    case ScheduleState::DISABLED:
      return "DISABLED";
    default:
      EnumParseOverflowContainer* overflowContainer = Aws::GetEnumOverflowContainer();
      if (overflowContainer) {
        return overflowContainer->RetrieveOverflow(static_cast<int>(enumValue));
      }

      return {};
  }
}

}  // namespace ScheduleStateMapper
}  // namespace Model
}  // namespace Scheduler
}  // namespace Aws
