﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#include <aws/core/Globals.h>
#include <aws/core/utils/EnumParseOverflowContainer.h>
#include <aws/core/utils/HashingUtils.h>
#include <aws/firehose/model/SSLMode.h>

using namespace Aws::Utils;

namespace Aws {
namespace Firehose {
namespace Model {
namespace SSLModeMapper {

static const int Disabled_HASH = HashingUtils::HashString("Disabled");
static const int Enabled_HASH = HashingUtils::HashString("Enabled");

SSLMode GetSSLModeForName(const Aws::String& name) {
  int hashCode = HashingUtils::HashString(name.c_str());
  if (hashCode == Disabled_HASH) {
    return SSLMode::Disabled;
  } else if (hashCode == Enabled_HASH) {
    return SSLMode::Enabled;
  }
  EnumParseOverflowContainer* overflowContainer = Aws::GetEnumOverflowContainer();
  if (overflowContainer) {
    overflowContainer->StoreOverflow(hashCode, name);
    return static_cast<SSLMode>(hashCode);
  }

  return SSLMode::NOT_SET;
}

Aws::String GetNameForSSLMode(SSLMode enumValue) {
  switch (enumValue) {
    case SSLMode::NOT_SET:
      return {};
    case SSLMode::Disabled:
      return "Disabled";
    case SSLMode::Enabled:
      return "Enabled";
    default:
      EnumParseOverflowContainer* overflowContainer = Aws::GetEnumOverflowContainer();
      if (overflowContainer) {
        return overflowContainer->RetrieveOverflow(static_cast<int>(enumValue));
      }

      return {};
  }
}

}  // namespace SSLModeMapper
}  // namespace Model
}  // namespace Firehose
}  // namespace Aws
