﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#include <aws/core/Globals.h>
#include <aws/core/utils/EnumParseOverflowContainer.h>
#include <aws/core/utils/HashingUtils.h>
#include <aws/iot/model/ConfigurationStatus.h>

using namespace Aws::Utils;

namespace Aws {
namespace IoT {
namespace Model {
namespace ConfigurationStatusMapper {

static const int HEALTHY_HASH = HashingUtils::HashString("HEALTHY");
static const int UNHEALTHY_HASH = HashingUtils::HashString("UNHEALTHY");

ConfigurationStatus GetConfigurationStatusForName(const Aws::String& name) {
  int hashCode = HashingUtils::HashString(name.c_str());
  if (hashCode == HEALTHY_HASH) {
    return ConfigurationStatus::HEALTHY;
  } else if (hashCode == UNHEALTHY_HASH) {
    return ConfigurationStatus::UNHEALTHY;
  }
  EnumParseOverflowContainer* overflowContainer = Aws::GetEnumOverflowContainer();
  if (overflowContainer) {
    overflowContainer->StoreOverflow(hashCode, name);
    return static_cast<ConfigurationStatus>(hashCode);
  }

  return ConfigurationStatus::NOT_SET;
}

Aws::String GetNameForConfigurationStatus(ConfigurationStatus enumValue) {
  switch (enumValue) {
    case ConfigurationStatus::NOT_SET:
      return {};
    case ConfigurationStatus::HEALTHY:
      return "HEALTHY";
    case ConfigurationStatus::UNHEALTHY:
      return "UNHEALTHY";
    default:
      EnumParseOverflowContainer* overflowContainer = Aws::GetEnumOverflowContainer();
      if (overflowContainer) {
        return overflowContainer->RetrieveOverflow(static_cast<int>(enumValue));
      }

      return {};
  }
}

}  // namespace ConfigurationStatusMapper
}  // namespace Model
}  // namespace IoT
}  // namespace Aws
