﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#include <aws/core/Globals.h>
#include <aws/core/utils/EnumParseOverflowContainer.h>
#include <aws/core/utils/HashingUtils.h>
#include <aws/mailmanager/model/ImportDataType.h>

using namespace Aws::Utils;

namespace Aws {
namespace MailManager {
namespace Model {
namespace ImportDataTypeMapper {

static const int CSV_HASH = HashingUtils::HashString("CSV");
static const int JSON_HASH = HashingUtils::HashString("JSON");

ImportDataType GetImportDataTypeForName(const Aws::String& name) {
  int hashCode = HashingUtils::HashString(name.c_str());
  if (hashCode == CSV_HASH) {
    return ImportDataType::CSV;
  } else if (hashCode == JSON_HASH) {
    return ImportDataType::JSON;
  }
  EnumParseOverflowContainer* overflowContainer = Aws::GetEnumOverflowContainer();
  if (overflowContainer) {
    overflowContainer->StoreOverflow(hashCode, name);
    return static_cast<ImportDataType>(hashCode);
  }

  return ImportDataType::NOT_SET;
}

Aws::String GetNameForImportDataType(ImportDataType enumValue) {
  switch (enumValue) {
    case ImportDataType::NOT_SET:
      return {};
    case ImportDataType::CSV:
      return "CSV";
    case ImportDataType::JSON:
      return "JSON";
    default:
      EnumParseOverflowContainer* overflowContainer = Aws::GetEnumOverflowContainer();
      if (overflowContainer) {
        return overflowContainer->RetrieveOverflow(static_cast<int>(enumValue));
      }

      return {};
  }
}

}  // namespace ImportDataTypeMapper
}  // namespace Model
}  // namespace MailManager
}  // namespace Aws
