﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#include <aws/core/Globals.h>
#include <aws/core/utils/EnumParseOverflowContainer.h>
#include <aws/core/utils/HashingUtils.h>
#include <aws/outposts/model/SubscriptionStatus.h>

using namespace Aws::Utils;

namespace Aws {
namespace Outposts {
namespace Model {
namespace SubscriptionStatusMapper {

static const int ACTIVE_HASH = HashingUtils::HashString("ACTIVE");
static const int INACTIVE_HASH = HashingUtils::HashString("INACTIVE");
static const int CANCELLED_HASH = HashingUtils::HashString("CANCELLED");

SubscriptionStatus GetSubscriptionStatusForName(const Aws::String& name) {
  int hashCode = HashingUtils::HashString(name.c_str());
  if (hashCode == ACTIVE_HASH) {
    return SubscriptionStatus::ACTIVE;
  } else if (hashCode == INACTIVE_HASH) {
    return SubscriptionStatus::INACTIVE;
  } else if (hashCode == CANCELLED_HASH) {
    return SubscriptionStatus::CANCELLED;
  }
  EnumParseOverflowContainer* overflowContainer = Aws::GetEnumOverflowContainer();
  if (overflowContainer) {
    overflowContainer->StoreOverflow(hashCode, name);
    return static_cast<SubscriptionStatus>(hashCode);
  }

  return SubscriptionStatus::NOT_SET;
}

Aws::String GetNameForSubscriptionStatus(SubscriptionStatus enumValue) {
  switch (enumValue) {
    case SubscriptionStatus::NOT_SET:
      return {};
    case SubscriptionStatus::ACTIVE:
      return "ACTIVE";
    case SubscriptionStatus::INACTIVE:
      return "INACTIVE";
    case SubscriptionStatus::CANCELLED:
      return "CANCELLED";
    default:
      EnumParseOverflowContainer* overflowContainer = Aws::GetEnumOverflowContainer();
      if (overflowContainer) {
        return overflowContainer->RetrieveOverflow(static_cast<int>(enumValue));
      }

      return {};
  }
}

}  // namespace SubscriptionStatusMapper
}  // namespace Model
}  // namespace Outposts
}  // namespace Aws
