﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#include <aws/core/Globals.h>
#include <aws/core/utils/EnumParseOverflowContainer.h>
#include <aws/core/utils/HashingUtils.h>
#include <aws/pinpoint/model/Frequency.h>

using namespace Aws::Utils;

namespace Aws {
namespace Pinpoint {
namespace Model {
namespace FrequencyMapper {

static const int ONCE_HASH = HashingUtils::HashString("ONCE");
static const int HOURLY_HASH = HashingUtils::HashString("HOURLY");
static const int DAILY_HASH = HashingUtils::HashString("DAILY");
static const int WEEKLY_HASH = HashingUtils::HashString("WEEKLY");
static const int MONTHLY_HASH = HashingUtils::HashString("MONTHLY");
static const int EVENT_HASH = HashingUtils::HashString("EVENT");
static const int IN_APP_EVENT_HASH = HashingUtils::HashString("IN_APP_EVENT");

Frequency GetFrequencyForName(const Aws::String& name) {
  int hashCode = HashingUtils::HashString(name.c_str());
  if (hashCode == ONCE_HASH) {
    return Frequency::ONCE;
  } else if (hashCode == HOURLY_HASH) {
    return Frequency::HOURLY;
  } else if (hashCode == DAILY_HASH) {
    return Frequency::DAILY;
  } else if (hashCode == WEEKLY_HASH) {
    return Frequency::WEEKLY;
  } else if (hashCode == MONTHLY_HASH) {
    return Frequency::MONTHLY;
  } else if (hashCode == EVENT_HASH) {
    return Frequency::EVENT;
  } else if (hashCode == IN_APP_EVENT_HASH) {
    return Frequency::IN_APP_EVENT;
  }
  EnumParseOverflowContainer* overflowContainer = Aws::GetEnumOverflowContainer();
  if (overflowContainer) {
    overflowContainer->StoreOverflow(hashCode, name);
    return static_cast<Frequency>(hashCode);
  }

  return Frequency::NOT_SET;
}

Aws::String GetNameForFrequency(Frequency enumValue) {
  switch (enumValue) {
    case Frequency::NOT_SET:
      return {};
    case Frequency::ONCE:
      return "ONCE";
    case Frequency::HOURLY:
      return "HOURLY";
    case Frequency::DAILY:
      return "DAILY";
    case Frequency::WEEKLY:
      return "WEEKLY";
    case Frequency::MONTHLY:
      return "MONTHLY";
    case Frequency::EVENT:
      return "EVENT";
    case Frequency::IN_APP_EVENT:
      return "IN_APP_EVENT";
    default:
      EnumParseOverflowContainer* overflowContainer = Aws::GetEnumOverflowContainer();
      if (overflowContainer) {
        return overflowContainer->RetrieveOverflow(static_cast<int>(enumValue));
      }

      return {};
  }
}

}  // namespace FrequencyMapper
}  // namespace Model
}  // namespace Pinpoint
}  // namespace Aws
