﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#include <aws/core/Globals.h>
#include <aws/core/utils/EnumParseOverflowContainer.h>
#include <aws/core/utils/HashingUtils.h>
#include <aws/rtbfabric/model/Protocol.h>

using namespace Aws::Utils;

namespace Aws {
namespace RTBFabric {
namespace Model {
namespace ProtocolMapper {

static const int HTTP_HASH = HashingUtils::HashString("HTTP");
static const int HTTPS_HASH = HashingUtils::HashString("HTTPS");

Protocol GetProtocolForName(const Aws::String& name) {
  int hashCode = HashingUtils::HashString(name.c_str());
  if (hashCode == HTTP_HASH) {
    return Protocol::HTTP;
  } else if (hashCode == HTTPS_HASH) {
    return Protocol::HTTPS;
  }
  EnumParseOverflowContainer* overflowContainer = Aws::GetEnumOverflowContainer();
  if (overflowContainer) {
    overflowContainer->StoreOverflow(hashCode, name);
    return static_cast<Protocol>(hashCode);
  }

  return Protocol::NOT_SET;
}

Aws::String GetNameForProtocol(Protocol enumValue) {
  switch (enumValue) {
    case Protocol::NOT_SET:
      return {};
    case Protocol::HTTP:
      return "HTTP";
    case Protocol::HTTPS:
      return "HTTPS";
    default:
      EnumParseOverflowContainer* overflowContainer = Aws::GetEnumOverflowContainer();
      if (overflowContainer) {
        return overflowContainer->RetrieveOverflow(static_cast<int>(enumValue));
      }

      return {};
  }
}

}  // namespace ProtocolMapper
}  // namespace Model
}  // namespace RTBFabric
}  // namespace Aws
