﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#include <aws/core/Globals.h>
#include <aws/core/utils/EnumParseOverflowContainer.h>
#include <aws/core/utils/HashingUtils.h>
#include <aws/textract/model/AutoUpdate.h>

using namespace Aws::Utils;

namespace Aws {
namespace Textract {
namespace Model {
namespace AutoUpdateMapper {

static const int ENABLED_HASH = HashingUtils::HashString("ENABLED");
static const int DISABLED_HASH = HashingUtils::HashString("DISABLED");

AutoUpdate GetAutoUpdateForName(const Aws::String& name) {
  int hashCode = HashingUtils::HashString(name.c_str());
  if (hashCode == ENABLED_HASH) {
    return AutoUpdate::ENABLED;
  } else if (hashCode == DISABLED_HASH) {
    return AutoUpdate::DISABLED;
  }
  EnumParseOverflowContainer* overflowContainer = Aws::GetEnumOverflowContainer();
  if (overflowContainer) {
    overflowContainer->StoreOverflow(hashCode, name);
    return static_cast<AutoUpdate>(hashCode);
  }

  return AutoUpdate::NOT_SET;
}

Aws::String GetNameForAutoUpdate(AutoUpdate enumValue) {
  switch (enumValue) {
    case AutoUpdate::NOT_SET:
      return {};
    case AutoUpdate::ENABLED:
      return "ENABLED";
    case AutoUpdate::DISABLED:
      return "DISABLED";
    default:
      EnumParseOverflowContainer* overflowContainer = Aws::GetEnumOverflowContainer();
      if (overflowContainer) {
        return overflowContainer->RetrieveOverflow(static_cast<int>(enumValue));
      }

      return {};
  }
}

}  // namespace AutoUpdateMapper
}  // namespace Model
}  // namespace Textract
}  // namespace Aws
