﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#include <aws/core/Globals.h>
#include <aws/core/utils/EnumParseOverflowContainer.h>
#include <aws/core/utils/HashingUtils.h>
#include <aws/observabilityadmin/model/Action.h>

using namespace Aws::Utils;

namespace Aws {
namespace ObservabilityAdmin {
namespace Model {
namespace ActionMapper {

static const int ALLOW_HASH = HashingUtils::HashString("ALLOW");
static const int BLOCK_HASH = HashingUtils::HashString("BLOCK");
static const int COUNT_HASH = HashingUtils::HashString("COUNT");
static const int CAPTCHA_HASH = HashingUtils::HashString("CAPTCHA");
static const int CHALLENGE_HASH = HashingUtils::HashString("CHALLENGE");
static const int EXCLUDED_AS_COUNT_HASH = HashingUtils::HashString("EXCLUDED_AS_COUNT");

Action GetActionForName(const Aws::String& name) {
  int hashCode = HashingUtils::HashString(name.c_str());
  if (hashCode == ALLOW_HASH) {
    return Action::ALLOW;
  } else if (hashCode == BLOCK_HASH) {
    return Action::BLOCK;
  } else if (hashCode == COUNT_HASH) {
    return Action::COUNT;
  } else if (hashCode == CAPTCHA_HASH) {
    return Action::CAPTCHA;
  } else if (hashCode == CHALLENGE_HASH) {
    return Action::CHALLENGE;
  } else if (hashCode == EXCLUDED_AS_COUNT_HASH) {
    return Action::EXCLUDED_AS_COUNT;
  }
  EnumParseOverflowContainer* overflowContainer = Aws::GetEnumOverflowContainer();
  if (overflowContainer) {
    overflowContainer->StoreOverflow(hashCode, name);
    return static_cast<Action>(hashCode);
  }

  return Action::NOT_SET;
}

Aws::String GetNameForAction(Action enumValue) {
  switch (enumValue) {
    case Action::NOT_SET:
      return {};
    case Action::ALLOW:
      return "ALLOW";
    case Action::BLOCK:
      return "BLOCK";
    case Action::COUNT:
      return "COUNT";
    case Action::CAPTCHA:
      return "CAPTCHA";
    case Action::CHALLENGE:
      return "CHALLENGE";
    case Action::EXCLUDED_AS_COUNT:
      return "EXCLUDED_AS_COUNT";
    default:
      EnumParseOverflowContainer* overflowContainer = Aws::GetEnumOverflowContainer();
      if (overflowContainer) {
        return overflowContainer->RetrieveOverflow(static_cast<int>(enumValue));
      }

      return {};
  }
}

}  // namespace ActionMapper
}  // namespace Model
}  // namespace ObservabilityAdmin
}  // namespace Aws
