﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#include <aws/core/Globals.h>
#include <aws/core/utils/EnumParseOverflowContainer.h>
#include <aws/core/utils/HashingUtils.h>
#include <aws/partnercentral-selling/model/TaskStatus.h>

using namespace Aws::Utils;

namespace Aws {
namespace PartnerCentralSelling {
namespace Model {
namespace TaskStatusMapper {

static const int IN_PROGRESS_HASH = HashingUtils::HashString("IN_PROGRESS");
static const int COMPLETE_HASH = HashingUtils::HashString("COMPLETE");
static const int FAILED_HASH = HashingUtils::HashString("FAILED");

TaskStatus GetTaskStatusForName(const Aws::String& name) {
  int hashCode = HashingUtils::HashString(name.c_str());
  if (hashCode == IN_PROGRESS_HASH) {
    return TaskStatus::IN_PROGRESS;
  } else if (hashCode == COMPLETE_HASH) {
    return TaskStatus::COMPLETE;
  } else if (hashCode == FAILED_HASH) {
    return TaskStatus::FAILED;
  }
  EnumParseOverflowContainer* overflowContainer = Aws::GetEnumOverflowContainer();
  if (overflowContainer) {
    overflowContainer->StoreOverflow(hashCode, name);
    return static_cast<TaskStatus>(hashCode);
  }

  return TaskStatus::NOT_SET;
}

Aws::String GetNameForTaskStatus(TaskStatus enumValue) {
  switch (enumValue) {
    case TaskStatus::NOT_SET:
      return {};
    case TaskStatus::IN_PROGRESS:
      return "IN_PROGRESS";
    case TaskStatus::COMPLETE:
      return "COMPLETE";
    case TaskStatus::FAILED:
      return "FAILED";
    default:
      EnumParseOverflowContainer* overflowContainer = Aws::GetEnumOverflowContainer();
      if (overflowContainer) {
        return overflowContainer->RetrieveOverflow(static_cast<int>(enumValue));
      }

      return {};
  }
}

}  // namespace TaskStatusMapper
}  // namespace Model
}  // namespace PartnerCentralSelling
}  // namespace Aws
