﻿#!/usr/bin/env python3

# Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
# SPDX-License-Identifier: Apache-2.0.
#
import argparse
import os
import stat
import subprocess
import platform
import random
import string


def parse_arguments():
    arg_map = {}

    parser = argparse.ArgumentParser(description="AWSNativeSDK Run all Integration Tests")
    parser.add_argument("--testDir", action="store")
    parser.add_argument("--serviceId", action="store")

    args = vars(parser.parse_args())
    arg_map["testDir"] = args["testDir"] or "./build"
    arg_map["serviceId"] = args["serviceId"] or ""

    return arg_map


def add_executable_bit(file):
    st = os.stat(file)
    os.chmod(file, st.st_mode | stat.S_IEXEC)


def main():
    arguments = parse_arguments()

    test_has_parent_dir = platform.system() != "Windows"
    exe_extension = ".exe" if platform.system() == "Windows" else ""

    all_tests = [
        "aws-cpp-sdk-core-integration-tests",
        "aws-cpp-sdk-transcribestreaming-integ-tests",
        "aws-cpp-sdk-dynamodb-unit-tests",
        "aws-cpp-sdk-dynamodb-integration-tests",
        "aws-cpp-sdk-sqs-integration-tests",
        "aws-cpp-sdk-sqs-unit-tests",
        "aws-cpp-sdk-sns-integration-tests",
        "aws-cpp-sdk-s3-integration-tests",
        "aws-cpp-sdk-s3-unit-tests",
        "aws-cpp-sdk-s3-crt-integration-tests",
        #"aws-cpp-sdk-s3control-integration-tests",
        "aws-cpp-sdk-lambda-integration-tests",
        "aws-cpp-sdk-cognitoidentity-integration-tests",
        #"aws-cpp-sdk-transfer-tests",
        "aws-cpp-sdk-s3-encryption-integration-tests",
        "aws-cpp-sdk-kinesis-integration-tests",
        "aws-cpp-sdk-logs-integration-tests",
        "aws-cpp-sdk-monitoring-integration-tests",
        "aws-cpp-sdk-elasticfilesystem-integration-tests",
        "aws-cpp-sdk-rds-integration-tests",
        "aws-cpp-sdk-ec2-integration-tests",
        "aws-cpp-sdk-bedrock-runtime-integration-tests"
    ]

    if arguments["serviceId"]:
        service_ids = arguments["serviceId"].split(",")
        test_list = []
        for test in all_tests:
            service = test.replace('aws-cpp-sdk-', '').replace('-integration-tests', '').replace('-unit-tests', '').replace('-tests', '')
            if service != "core":
                for service_id in service_ids:
                    if service_id == service:
                        test_list.append(test)
                        break
            else:
                test_list.append(test)
    else:
        test_list = all_tests.copy()

    # check for existence of these binaries before adding them to tests
    # as they will not always be present
    cmake_dependent_tests = [
        "aws-cpp-sdk-s3-crt-memory-checked-integration-tests"
    ]

    for test in cmake_dependent_tests:
        test_exe = os.path.join(arguments["testDir"], test if test_has_parent_dir else "", test) + exe_extension
        if os.path.isfile(test_exe):
            test_list.append("aws-cpp-sdk-s3-crt-memory-checked-integration-tests")

    random.shuffle(test_list)

    for testName in test_list:
        test_exe = os.path.join(arguments["testDir"], testName if test_has_parent_dir else "", testName) + exe_extension
        # when build with BUILD_ONLY, not all test binaries will be generated.
        if not os.path.isfile(test_exe):
            print("Test: \"{}\" doesn't exist, failing test run.".format(test_exe))
            exit(1)
        prefix = "--aws_resource_prefix=" + ''.join(
            random.choice(string.ascii_lowercase + string.digits) for _ in range(8))
        print("testExe = " + test_exe)
        print("prefix = " + prefix)
        gtest_brief = "--gtest_brief=1"
        add_executable_bit(test_exe)
        subprocess.check_call([test_exe, prefix, gtest_brief], timeout=20*60)


# Run from powershell; make sure msbuild is in PATH environment variable
if __name__ == "__main__":
    main()
