﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#include <aws/core/Globals.h>
#include <aws/core/utils/EnumParseOverflowContainer.h>
#include <aws/core/utils/HashingUtils.h>
#include <aws/detective/model/State.h>

using namespace Aws::Utils;

namespace Aws {
namespace Detective {
namespace Model {
namespace StateMapper {

static const int ACTIVE_HASH = HashingUtils::HashString("ACTIVE");
static const int ARCHIVED_HASH = HashingUtils::HashString("ARCHIVED");

State GetStateForName(const Aws::String& name) {
  int hashCode = HashingUtils::HashString(name.c_str());
  if (hashCode == ACTIVE_HASH) {
    return State::ACTIVE;
  } else if (hashCode == ARCHIVED_HASH) {
    return State::ARCHIVED;
  }
  EnumParseOverflowContainer* overflowContainer = Aws::GetEnumOverflowContainer();
  if (overflowContainer) {
    overflowContainer->StoreOverflow(hashCode, name);
    return static_cast<State>(hashCode);
  }

  return State::NOT_SET;
}

Aws::String GetNameForState(State enumValue) {
  switch (enumValue) {
    case State::NOT_SET:
      return {};
    case State::ACTIVE:
      return "ACTIVE";
    case State::ARCHIVED:
      return "ARCHIVED";
    default:
      EnumParseOverflowContainer* overflowContainer = Aws::GetEnumOverflowContainer();
      if (overflowContainer) {
        return overflowContainer->RetrieveOverflow(static_cast<int>(enumValue));
      }

      return {};
  }
}

}  // namespace StateMapper
}  // namespace Model
}  // namespace Detective
}  // namespace Aws
