﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#include <aws/core/Globals.h>
#include <aws/core/utils/EnumParseOverflowContainer.h>
#include <aws/core/utils/HashingUtils.h>
#include <aws/dynamodb/model/TableStatus.h>

using namespace Aws::Utils;

namespace Aws {
namespace DynamoDB {
namespace Model {
namespace TableStatusMapper {

static const int CREATING_HASH = HashingUtils::HashString("CREATING");
static const int UPDATING_HASH = HashingUtils::HashString("UPDATING");
static const int DELETING_HASH = HashingUtils::HashString("DELETING");
static const int ACTIVE_HASH = HashingUtils::HashString("ACTIVE");
static const int INACCESSIBLE_ENCRYPTION_CREDENTIALS_HASH = HashingUtils::HashString("INACCESSIBLE_ENCRYPTION_CREDENTIALS");
static const int ARCHIVING_HASH = HashingUtils::HashString("ARCHIVING");
static const int ARCHIVED_HASH = HashingUtils::HashString("ARCHIVED");
static const int REPLICATION_NOT_AUTHORIZED_HASH = HashingUtils::HashString("REPLICATION_NOT_AUTHORIZED");

TableStatus GetTableStatusForName(const Aws::String& name) {
  int hashCode = HashingUtils::HashString(name.c_str());
  if (hashCode == CREATING_HASH) {
    return TableStatus::CREATING;
  } else if (hashCode == UPDATING_HASH) {
    return TableStatus::UPDATING;
  } else if (hashCode == DELETING_HASH) {
    return TableStatus::DELETING;
  } else if (hashCode == ACTIVE_HASH) {
    return TableStatus::ACTIVE;
  } else if (hashCode == INACCESSIBLE_ENCRYPTION_CREDENTIALS_HASH) {
    return TableStatus::INACCESSIBLE_ENCRYPTION_CREDENTIALS;
  } else if (hashCode == ARCHIVING_HASH) {
    return TableStatus::ARCHIVING;
  } else if (hashCode == ARCHIVED_HASH) {
    return TableStatus::ARCHIVED;
  } else if (hashCode == REPLICATION_NOT_AUTHORIZED_HASH) {
    return TableStatus::REPLICATION_NOT_AUTHORIZED;
  }
  EnumParseOverflowContainer* overflowContainer = Aws::GetEnumOverflowContainer();
  if (overflowContainer) {
    overflowContainer->StoreOverflow(hashCode, name);
    return static_cast<TableStatus>(hashCode);
  }

  return TableStatus::NOT_SET;
}

Aws::String GetNameForTableStatus(TableStatus enumValue) {
  switch (enumValue) {
    case TableStatus::NOT_SET:
      return {};
    case TableStatus::CREATING:
      return "CREATING";
    case TableStatus::UPDATING:
      return "UPDATING";
    case TableStatus::DELETING:
      return "DELETING";
    case TableStatus::ACTIVE:
      return "ACTIVE";
    case TableStatus::INACCESSIBLE_ENCRYPTION_CREDENTIALS:
      return "INACCESSIBLE_ENCRYPTION_CREDENTIALS";
    case TableStatus::ARCHIVING:
      return "ARCHIVING";
    case TableStatus::ARCHIVED:
      return "ARCHIVED";
    case TableStatus::REPLICATION_NOT_AUTHORIZED:
      return "REPLICATION_NOT_AUTHORIZED";
    default:
      EnumParseOverflowContainer* overflowContainer = Aws::GetEnumOverflowContainer();
      if (overflowContainer) {
        return overflowContainer->RetrieveOverflow(static_cast<int>(enumValue));
      }

      return {};
  }
}

}  // namespace TableStatusMapper
}  // namespace Model
}  // namespace DynamoDB
}  // namespace Aws
