﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#include <aws/core/utils/json/JsonSerializer.h>
#include <aws/dynamodb/model/UpdateItemRequest.h>

#include <utility>

using namespace Aws::DynamoDB::Model;
using namespace Aws::Utils::Json;
using namespace Aws::Utils;

Aws::String UpdateItemRequest::SerializePayload() const {
  JsonValue payload;

  if (m_tableNameHasBeenSet) {
    payload.WithString("TableName", m_tableName);
  }

  if (m_keyHasBeenSet) {
    JsonValue keyJsonMap;
    for (auto& keyItem : m_key) {
      keyJsonMap.WithObject(keyItem.first, keyItem.second.Jsonize());
    }
    payload.WithObject("Key", std::move(keyJsonMap));
  }

  if (m_attributeUpdatesHasBeenSet) {
    JsonValue attributeUpdatesJsonMap;
    for (auto& attributeUpdatesItem : m_attributeUpdates) {
      attributeUpdatesJsonMap.WithObject(attributeUpdatesItem.first, attributeUpdatesItem.second.Jsonize());
    }
    payload.WithObject("AttributeUpdates", std::move(attributeUpdatesJsonMap));
  }

  if (m_expectedHasBeenSet) {
    JsonValue expectedJsonMap;
    for (auto& expectedItem : m_expected) {
      expectedJsonMap.WithObject(expectedItem.first, expectedItem.second.Jsonize());
    }
    payload.WithObject("Expected", std::move(expectedJsonMap));
  }

  if (m_conditionalOperatorHasBeenSet) {
    payload.WithString("ConditionalOperator", ConditionalOperatorMapper::GetNameForConditionalOperator(m_conditionalOperator));
  }

  if (m_returnValuesHasBeenSet) {
    payload.WithString("ReturnValues", ReturnValueMapper::GetNameForReturnValue(m_returnValues));
  }

  if (m_returnConsumedCapacityHasBeenSet) {
    payload.WithString("ReturnConsumedCapacity", ReturnConsumedCapacityMapper::GetNameForReturnConsumedCapacity(m_returnConsumedCapacity));
  }

  if (m_returnItemCollectionMetricsHasBeenSet) {
    payload.WithString("ReturnItemCollectionMetrics",
                       ReturnItemCollectionMetricsMapper::GetNameForReturnItemCollectionMetrics(m_returnItemCollectionMetrics));
  }

  if (m_updateExpressionHasBeenSet) {
    payload.WithString("UpdateExpression", m_updateExpression);
  }

  if (m_conditionExpressionHasBeenSet) {
    payload.WithString("ConditionExpression", m_conditionExpression);
  }

  if (m_expressionAttributeNamesHasBeenSet) {
    JsonValue expressionAttributeNamesJsonMap;
    for (auto& expressionAttributeNamesItem : m_expressionAttributeNames) {
      expressionAttributeNamesJsonMap.WithString(expressionAttributeNamesItem.first, expressionAttributeNamesItem.second);
    }
    payload.WithObject("ExpressionAttributeNames", std::move(expressionAttributeNamesJsonMap));
  }

  if (m_expressionAttributeValuesHasBeenSet) {
    JsonValue expressionAttributeValuesJsonMap;
    for (auto& expressionAttributeValuesItem : m_expressionAttributeValues) {
      expressionAttributeValuesJsonMap.WithObject(expressionAttributeValuesItem.first, expressionAttributeValuesItem.second.Jsonize());
    }
    payload.WithObject("ExpressionAttributeValues", std::move(expressionAttributeValuesJsonMap));
  }

  if (m_returnValuesOnConditionCheckFailureHasBeenSet) {
    payload.WithString(
        "ReturnValuesOnConditionCheckFailure",
        ReturnValuesOnConditionCheckFailureMapper::GetNameForReturnValuesOnConditionCheckFailure(m_returnValuesOnConditionCheckFailure));
  }

  return payload.View().WriteReadable();
}

Aws::Http::HeaderValueCollection UpdateItemRequest::GetRequestSpecificHeaders() const {
  Aws::Http::HeaderValueCollection headers;
  headers.insert(Aws::Http::HeaderValuePair("X-Amz-Target", "DynamoDB_20120810.UpdateItem"));
  return headers;
}

UpdateItemRequest::EndpointParameters UpdateItemRequest::GetEndpointContextParams() const {
  EndpointParameters parameters;
  // Operation context parameters
  if (TableNameHasBeenSet()) {
    parameters.emplace_back(Aws::String("ResourceArn"), this->GetTableName(),
                            Aws::Endpoint::EndpointParameter::ParameterOrigin::OPERATION_CONTEXT);
  }
  return parameters;
}
