﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#include <aws/core/Globals.h>
#include <aws/core/utils/EnumParseOverflowContainer.h>
#include <aws/core/utils/HashingUtils.h>
#include <aws/eventbridge/model/RuleState.h>

using namespace Aws::Utils;

namespace Aws {
namespace EventBridge {
namespace Model {
namespace RuleStateMapper {

static const int ENABLED_HASH = HashingUtils::HashString("ENABLED");
static const int DISABLED_HASH = HashingUtils::HashString("DISABLED");
static const int ENABLED_WITH_ALL_CLOUDTRAIL_MANAGEMENT_EVENTS_HASH =
    HashingUtils::HashString("ENABLED_WITH_ALL_CLOUDTRAIL_MANAGEMENT_EVENTS");

RuleState GetRuleStateForName(const Aws::String& name) {
  int hashCode = HashingUtils::HashString(name.c_str());
  if (hashCode == ENABLED_HASH) {
    return RuleState::ENABLED;
  } else if (hashCode == DISABLED_HASH) {
    return RuleState::DISABLED;
  } else if (hashCode == ENABLED_WITH_ALL_CLOUDTRAIL_MANAGEMENT_EVENTS_HASH) {
    return RuleState::ENABLED_WITH_ALL_CLOUDTRAIL_MANAGEMENT_EVENTS;
  }
  EnumParseOverflowContainer* overflowContainer = Aws::GetEnumOverflowContainer();
  if (overflowContainer) {
    overflowContainer->StoreOverflow(hashCode, name);
    return static_cast<RuleState>(hashCode);
  }

  return RuleState::NOT_SET;
}

Aws::String GetNameForRuleState(RuleState enumValue) {
  switch (enumValue) {
    case RuleState::NOT_SET:
      return {};
    case RuleState::ENABLED:
      return "ENABLED";
    case RuleState::DISABLED:
      return "DISABLED";
    case RuleState::ENABLED_WITH_ALL_CLOUDTRAIL_MANAGEMENT_EVENTS:
      return "ENABLED_WITH_ALL_CLOUDTRAIL_MANAGEMENT_EVENTS";
    default:
      EnumParseOverflowContainer* overflowContainer = Aws::GetEnumOverflowContainer();
      if (overflowContainer) {
        return overflowContainer->RetrieveOverflow(static_cast<int>(enumValue));
      }

      return {};
  }
}

}  // namespace RuleStateMapper
}  // namespace Model
}  // namespace EventBridge
}  // namespace Aws
