﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#include <aws/core/Globals.h>
#include <aws/core/utils/EnumParseOverflowContainer.h>
#include <aws/core/utils/HashingUtils.h>
#include <aws/glacier/model/EncryptionType.h>

using namespace Aws::Utils;

namespace Aws {
namespace Glacier {
namespace Model {
namespace EncryptionTypeMapper {

static const int aws_kms_HASH = HashingUtils::HashString("aws:kms");
static const int AES256_HASH = HashingUtils::HashString("AES256");

EncryptionType GetEncryptionTypeForName(const Aws::String& name) {
  int hashCode = HashingUtils::HashString(name.c_str());
  if (hashCode == aws_kms_HASH) {
    return EncryptionType::aws_kms;
  } else if (hashCode == AES256_HASH) {
    return EncryptionType::AES256;
  }
  EnumParseOverflowContainer* overflowContainer = Aws::GetEnumOverflowContainer();
  if (overflowContainer) {
    overflowContainer->StoreOverflow(hashCode, name);
    return static_cast<EncryptionType>(hashCode);
  }

  return EncryptionType::NOT_SET;
}

Aws::String GetNameForEncryptionType(EncryptionType enumValue) {
  switch (enumValue) {
    case EncryptionType::NOT_SET:
      return {};
    case EncryptionType::aws_kms:
      return "aws:kms";
    case EncryptionType::AES256:
      return "AES256";
    default:
      EnumParseOverflowContainer* overflowContainer = Aws::GetEnumOverflowContainer();
      if (overflowContainer) {
        return overflowContainer->RetrieveOverflow(static_cast<int>(enumValue));
      }

      return {};
  }
}

}  // namespace EncryptionTypeMapper
}  // namespace Model
}  // namespace Glacier
}  // namespace Aws
