﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#include <aws/core/Globals.h>
#include <aws/core/utils/EnumParseOverflowContainer.h>
#include <aws/core/utils/HashingUtils.h>
#include <aws/greengrassv2/model/DeploymentStatus.h>

using namespace Aws::Utils;

namespace Aws {
namespace GreengrassV2 {
namespace Model {
namespace DeploymentStatusMapper {

static const int ACTIVE_HASH = HashingUtils::HashString("ACTIVE");
static const int COMPLETED_HASH = HashingUtils::HashString("COMPLETED");
static const int CANCELED_HASH = HashingUtils::HashString("CANCELED");
static const int FAILED_HASH = HashingUtils::HashString("FAILED");
static const int INACTIVE_HASH = HashingUtils::HashString("INACTIVE");

DeploymentStatus GetDeploymentStatusForName(const Aws::String& name) {
  int hashCode = HashingUtils::HashString(name.c_str());
  if (hashCode == ACTIVE_HASH) {
    return DeploymentStatus::ACTIVE;
  } else if (hashCode == COMPLETED_HASH) {
    return DeploymentStatus::COMPLETED;
  } else if (hashCode == CANCELED_HASH) {
    return DeploymentStatus::CANCELED;
  } else if (hashCode == FAILED_HASH) {
    return DeploymentStatus::FAILED;
  } else if (hashCode == INACTIVE_HASH) {
    return DeploymentStatus::INACTIVE;
  }
  EnumParseOverflowContainer* overflowContainer = Aws::GetEnumOverflowContainer();
  if (overflowContainer) {
    overflowContainer->StoreOverflow(hashCode, name);
    return static_cast<DeploymentStatus>(hashCode);
  }

  return DeploymentStatus::NOT_SET;
}

Aws::String GetNameForDeploymentStatus(DeploymentStatus enumValue) {
  switch (enumValue) {
    case DeploymentStatus::NOT_SET:
      return {};
    case DeploymentStatus::ACTIVE:
      return "ACTIVE";
    case DeploymentStatus::COMPLETED:
      return "COMPLETED";
    case DeploymentStatus::CANCELED:
      return "CANCELED";
    case DeploymentStatus::FAILED:
      return "FAILED";
    case DeploymentStatus::INACTIVE:
      return "INACTIVE";
    default:
      EnumParseOverflowContainer* overflowContainer = Aws::GetEnumOverflowContainer();
      if (overflowContainer) {
        return overflowContainer->RetrieveOverflow(static_cast<int>(enumValue));
      }

      return {};
  }
}

}  // namespace DeploymentStatusMapper
}  // namespace Model
}  // namespace GreengrassV2
}  // namespace Aws
