﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#include <aws/core/Globals.h>
#include <aws/core/utils/EnumParseOverflowContainer.h>
#include <aws/core/utils/HashingUtils.h>
#include <aws/lightsail/model/IpAddressType.h>

using namespace Aws::Utils;

namespace Aws {
namespace Lightsail {
namespace Model {
namespace IpAddressTypeMapper {

static const int dualstack_HASH = HashingUtils::HashString("dualstack");
static const int ipv4_HASH = HashingUtils::HashString("ipv4");
static const int ipv6_HASH = HashingUtils::HashString("ipv6");

IpAddressType GetIpAddressTypeForName(const Aws::String& name) {
  int hashCode = HashingUtils::HashString(name.c_str());
  if (hashCode == dualstack_HASH) {
    return IpAddressType::dualstack;
  } else if (hashCode == ipv4_HASH) {
    return IpAddressType::ipv4;
  } else if (hashCode == ipv6_HASH) {
    return IpAddressType::ipv6;
  }
  EnumParseOverflowContainer* overflowContainer = Aws::GetEnumOverflowContainer();
  if (overflowContainer) {
    overflowContainer->StoreOverflow(hashCode, name);
    return static_cast<IpAddressType>(hashCode);
  }

  return IpAddressType::NOT_SET;
}

Aws::String GetNameForIpAddressType(IpAddressType enumValue) {
  switch (enumValue) {
    case IpAddressType::NOT_SET:
      return {};
    case IpAddressType::dualstack:
      return "dualstack";
    case IpAddressType::ipv4:
      return "ipv4";
    case IpAddressType::ipv6:
      return "ipv6";
    default:
      EnumParseOverflowContainer* overflowContainer = Aws::GetEnumOverflowContainer();
      if (overflowContainer) {
        return overflowContainer->RetrieveOverflow(static_cast<int>(enumValue));
      }

      return {};
  }
}

}  // namespace IpAddressTypeMapper
}  // namespace Model
}  // namespace Lightsail
}  // namespace Aws
