﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#include <aws/core/Globals.h>
#include <aws/core/utils/EnumParseOverflowContainer.h>
#include <aws/core/utils/HashingUtils.h>
#include <aws/qbusiness/model/PluginState.h>

using namespace Aws::Utils;

namespace Aws {
namespace QBusiness {
namespace Model {
namespace PluginStateMapper {

static const int ENABLED_HASH = HashingUtils::HashString("ENABLED");
static const int DISABLED_HASH = HashingUtils::HashString("DISABLED");

PluginState GetPluginStateForName(const Aws::String& name) {
  int hashCode = HashingUtils::HashString(name.c_str());
  if (hashCode == ENABLED_HASH) {
    return PluginState::ENABLED;
  } else if (hashCode == DISABLED_HASH) {
    return PluginState::DISABLED;
  }
  EnumParseOverflowContainer* overflowContainer = Aws::GetEnumOverflowContainer();
  if (overflowContainer) {
    overflowContainer->StoreOverflow(hashCode, name);
    return static_cast<PluginState>(hashCode);
  }

  return PluginState::NOT_SET;
}

Aws::String GetNameForPluginState(PluginState enumValue) {
  switch (enumValue) {
    case PluginState::NOT_SET:
      return {};
    case PluginState::ENABLED:
      return "ENABLED";
    case PluginState::DISABLED:
      return "DISABLED";
    default:
      EnumParseOverflowContainer* overflowContainer = Aws::GetEnumOverflowContainer();
      if (overflowContainer) {
        return overflowContainer->RetrieveOverflow(static_cast<int>(enumValue));
      }

      return {};
  }
}

}  // namespace PluginStateMapper
}  // namespace Model
}  // namespace QBusiness
}  // namespace Aws
