﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#include <aws/core/Globals.h>
#include <aws/core/utils/EnumParseOverflowContainer.h>
#include <aws/core/utils/HashingUtils.h>
#include <aws/qbusiness/model/ReadAccessType.h>

using namespace Aws::Utils;

namespace Aws {
namespace QBusiness {
namespace Model {
namespace ReadAccessTypeMapper {

static const int ALLOW_HASH = HashingUtils::HashString("ALLOW");
static const int DENY_HASH = HashingUtils::HashString("DENY");

ReadAccessType GetReadAccessTypeForName(const Aws::String& name) {
  int hashCode = HashingUtils::HashString(name.c_str());
  if (hashCode == ALLOW_HASH) {
    return ReadAccessType::ALLOW;
  } else if (hashCode == DENY_HASH) {
    return ReadAccessType::DENY;
  }
  EnumParseOverflowContainer* overflowContainer = Aws::GetEnumOverflowContainer();
  if (overflowContainer) {
    overflowContainer->StoreOverflow(hashCode, name);
    return static_cast<ReadAccessType>(hashCode);
  }

  return ReadAccessType::NOT_SET;
}

Aws::String GetNameForReadAccessType(ReadAccessType enumValue) {
  switch (enumValue) {
    case ReadAccessType::NOT_SET:
      return {};
    case ReadAccessType::ALLOW:
      return "ALLOW";
    case ReadAccessType::DENY:
      return "DENY";
    default:
      EnumParseOverflowContainer* overflowContainer = Aws::GetEnumOverflowContainer();
      if (overflowContainer) {
        return overflowContainer->RetrieveOverflow(static_cast<int>(enumValue));
      }

      return {};
  }
}

}  // namespace ReadAccessTypeMapper
}  // namespace Model
}  // namespace QBusiness
}  // namespace Aws
