﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#include <aws/core/Globals.h>
#include <aws/core/utils/EnumParseOverflowContainer.h>
#include <aws/core/utils/HashingUtils.h>
#include <aws/s3control/model/ExpirationStatus.h>

using namespace Aws::Utils;

namespace Aws {
namespace S3Control {
namespace Model {
namespace ExpirationStatusMapper {

static const int Enabled_HASH = HashingUtils::HashString("Enabled");
static const int Disabled_HASH = HashingUtils::HashString("Disabled");

ExpirationStatus GetExpirationStatusForName(const Aws::String& name) {
  int hashCode = HashingUtils::HashString(name.c_str());
  if (hashCode == Enabled_HASH) {
    return ExpirationStatus::Enabled;
  } else if (hashCode == Disabled_HASH) {
    return ExpirationStatus::Disabled;
  }
  EnumParseOverflowContainer* overflowContainer = Aws::GetEnumOverflowContainer();
  if (overflowContainer) {
    overflowContainer->StoreOverflow(hashCode, name);
    return static_cast<ExpirationStatus>(hashCode);
  }

  return ExpirationStatus::NOT_SET;
}

Aws::String GetNameForExpirationStatus(ExpirationStatus enumValue) {
  switch (enumValue) {
    case ExpirationStatus::NOT_SET:
      return {};
    case ExpirationStatus::Enabled:
      return "Enabled";
    case ExpirationStatus::Disabled:
      return "Disabled";
    default:
      EnumParseOverflowContainer* overflowContainer = Aws::GetEnumOverflowContainer();
      if (overflowContainer) {
        return overflowContainer->RetrieveOverflow(static_cast<int>(enumValue));
      }

      return {};
  }
}

}  // namespace ExpirationStatusMapper
}  // namespace Model
}  // namespace S3Control
}  // namespace Aws
