﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#include <aws/core/client/AWSError.h>
#include <aws/core/utils/HashingUtils.h>
#include <aws/s3tables/S3TablesErrors.h>

using namespace Aws::Client;
using namespace Aws::Utils;
using namespace Aws::S3Tables;

namespace Aws {
namespace S3Tables {
namespace S3TablesErrorMapper {

static const int CONFLICT_HASH = HashingUtils::HashString("ConflictException");
static const int NOT_FOUND_HASH = HashingUtils::HashString("NotFoundException");
static const int FORBIDDEN_HASH = HashingUtils::HashString("ForbiddenException");
static const int TOO_MANY_REQUESTS_HASH = HashingUtils::HashString("TooManyRequestsException");
static const int INTERNAL_SERVER_ERROR_HASH = HashingUtils::HashString("InternalServerErrorException");
static const int BAD_REQUEST_HASH = HashingUtils::HashString("BadRequestException");
static const int METHOD_NOT_ALLOWED_HASH = HashingUtils::HashString("MethodNotAllowedException");

AWSError<CoreErrors> GetErrorForName(const char* errorName) {
  int hashCode = HashingUtils::HashString(errorName);

  if (hashCode == CONFLICT_HASH) {
    return AWSError<CoreErrors>(static_cast<CoreErrors>(S3TablesErrors::CONFLICT), RetryableType::NOT_RETRYABLE);
  } else if (hashCode == NOT_FOUND_HASH) {
    return AWSError<CoreErrors>(static_cast<CoreErrors>(S3TablesErrors::NOT_FOUND), RetryableType::NOT_RETRYABLE);
  } else if (hashCode == FORBIDDEN_HASH) {
    return AWSError<CoreErrors>(static_cast<CoreErrors>(S3TablesErrors::FORBIDDEN), RetryableType::NOT_RETRYABLE);
  } else if (hashCode == TOO_MANY_REQUESTS_HASH) {
    return AWSError<CoreErrors>(static_cast<CoreErrors>(S3TablesErrors::TOO_MANY_REQUESTS), RetryableType::RETRYABLE);
  } else if (hashCode == INTERNAL_SERVER_ERROR_HASH) {
    return AWSError<CoreErrors>(static_cast<CoreErrors>(S3TablesErrors::INTERNAL_SERVER_ERROR), RetryableType::RETRYABLE);
  } else if (hashCode == BAD_REQUEST_HASH) {
    return AWSError<CoreErrors>(static_cast<CoreErrors>(S3TablesErrors::BAD_REQUEST), RetryableType::NOT_RETRYABLE);
  } else if (hashCode == METHOD_NOT_ALLOWED_HASH) {
    return AWSError<CoreErrors>(static_cast<CoreErrors>(S3TablesErrors::METHOD_NOT_ALLOWED), RetryableType::NOT_RETRYABLE);
  }
  return AWSError<CoreErrors>(CoreErrors::UNKNOWN, false);
}

}  // namespace S3TablesErrorMapper
}  // namespace S3Tables
}  // namespace Aws
