#include <aws/s3-encryption/HKDF.h>
#include <aws/core/utils/crypto/CryptoBuf.h>
#include <aws/crt/crypto/HKDF.h>

namespace Aws {
namespace S3Encryption {

const size_t ENCRYPTION_KEY_INFO_LEN = 11;
const size_t COMMITMENT_KEY_INFO_LEN = 11;
const char *ENCRYPTION_KEY_INFO = "\x00\x73" "DERIVEKEY";
const char *COMMITMENT_KEY_INFO = "\x00\x73" "COMMITKEY";

bool derive_key(const Aws::Utils::CryptoBuffer &data_key, const Aws::Utils::CryptoBuffer &message_id, Aws::Crt::ByteCursor & info,
                           Aws::Utils::CryptoBuffer &output) {
  auto data_key_cursor = Aws::Crt::ByteCursorFromArray(data_key.GetUnderlyingData(), data_key.GetLength());
  auto message_id_cursor = Aws::Crt::ByteCursorFromArray(message_id.GetUnderlyingData(), message_id.GetLength());
  auto out = Aws::Crt::ByteBufFromEmptyArray(output.GetUnderlyingData(), output.GetLength());

  return Aws::Crt::Crypto::DeriveSHA512HMACHKDF(Aws::Crt::ApiAllocator(),
    data_key_cursor,
    message_id_cursor,
    info,
    out, output.GetLength());
}

bool derive_encryption_key(const Aws::Utils::CryptoBuffer &data_key, const Aws::Utils::CryptoBuffer &message_id,
                           Aws::Utils::CryptoBuffer &output) {

  //= ../specification/s3-encryption/key-derivation.md#hkdf-operation
  //# - The input keying material MUST be the plaintext data key (PDK) generated by the key provider.

  //= ../specification/s3-encryption/key-derivation.md#hkdf-operation
  //# - The length of the input keying material MUST equal the key derivation input length specified by the algorithm suite commit key derivation setting.

  //= ../specification/s3-encryption/key-derivation.md#hkdf-operation
  //# - The salt MUST be the Message ID with the length defined in the algorithm suite.

  //= ../specification/s3-encryption/key-derivation.md#hkdf-operation
  //# - The input info MUST be a concatenation of the algorithm suite ID as bytes followed by the string DERIVEKEY as UTF8 encoded bytes.

  //= ../specification/s3-encryption/key-derivation.md#hkdf-operation
  //# - The length of the output keying material MUST equal the encryption key length specified by the algorithm suite encryption settings.

  //= ../specification/s3-encryption/key-derivation.md#hkdf-operation
  //# - The hash function MUST be specified by the algorithm suite commitment settings.

  //= ../specification/s3-encryption/key-derivation.md#hkdf-operation
  //= type=implication
  //# - The DEK input pseudorandom key MUST be the output from the extract step.
  auto info_cursor = Aws::Crt::ByteCursorFromArray((const uint8_t *)ENCRYPTION_KEY_INFO, ENCRYPTION_KEY_INFO_LEN);
  return derive_key(data_key, message_id, info_cursor, output);
}

//= ../specification/s3-encryption/encryption.md#alg-aes-256-gcm-hkdf-sha512-commit-key
//= type=implication
//# The client MUST use HKDF to derive the key commitment value and the derived encrypting key as described in [Key Derivation](key-derivation.md).
bool derive_commitment_key(const Aws::Utils::CryptoBuffer &data_key, const Aws::Utils::CryptoBuffer &message_id,
                           Aws::Utils::CryptoBuffer &output) {
  //= ../specification/s3-encryption/key-derivation.md#hkdf-operation
  //# - The input keying material MUST be the plaintext data key (PDK) generated by the key provider.

  //= ../specification/s3-encryption/key-derivation.md#hkdf-operation
  //# - The length of the input keying material MUST equal the key derivation input length specified by the algorithm suite commit key derivation setting.

  //= ../specification/s3-encryption/key-derivation.md#hkdf-operation
  //# - The salt MUST be the Message ID with the length defined in the algorithm suite.

  //= ../specification/s3-encryption/key-derivation.md#hkdf-operation
  //# - The input info MUST be a concatenation of the algorithm suite ID as bytes followed by the string COMMITKEY as UTF8 encoded bytes.

  //= ../specification/s3-encryption/key-derivation.md#hkdf-operation
  //# - The length of the output keying material MUST equal the commit key length specified by the supported algorithm suites.

  //= ../specification/s3-encryption/key-derivation.md#hkdf-operation
  //# - The hash function MUST be specified by the algorithm suite commitment settings.

  //= ../specification/s3-encryption/key-derivation.md#hkdf-operation
  //= type=implication
  //# - The CK input pseudorandom key MUST be the output from the extract step.
  auto info_cursor = Aws::Crt::ByteCursorFromArray((const uint8_t *)COMMITMENT_KEY_INFO, COMMITMENT_KEY_INFO_LEN);
  return derive_key(data_key, message_id, info_cursor, output);
}

// The cal-c people promise to surface a "real" implementation of this at some point.
bool constant_time_equal(const Aws::Utils::CryptoBuffer& x, const Aws::Utils::CryptoBuffer& y)
{
  // If lengths don't match, they're not equal
  if (x.GetLength() != y.GetLength()) {
    return false;
  }
  
  const uint8_t* x_data = x.GetUnderlyingData();
  const uint8_t* y_data = y.GetUnderlyingData();
  
  volatile uint8_t val = 0;
  for (size_t i=0; i<x.GetLength(); ++i) {
    val |= x_data[i] ^ y_data[i];
  }
  return val == 0;
}

}  // namespace S3Encryption
}  // namespace Aws
