// SPDX-FileCopyrightText: 2018 Volker Krause <vkrause@kde.org>
// SPDX-FileCopyrightText: 2022 Carl Schwan <carl@carlschwan.eu>
// SPDX-License-Identifier: LGPL-2.0-or-later

import QtQml
import QtQuick
import QtQuick.Controls as QQC2
import QtPositioning
import org.kde.kirigami as Kirigami
import org.kde.kirigamiaddons.formcard as FormCard
import org.kde.i18n.localeData
import org.kde.notification as KNotification
import org.kde.kpublictransport.onboard
import org.kde.kpublictransport.ui as KPublicTransport
import org.kde.itinerary

FormCard.FormCardPage {
    id: root

    title: i18n("Settings")

    property Component ptBackendPage: Component {
        id: ptBackendPage
        KPublicTransport.BackendPage {
            publicTransportManager: LiveDataManager.publicTransportManager
            actions: [ onlineUpdateAction ]
        }
    }

    FormCard.FormHeader {
        title: i18n("Localization")
        visible: localizationCard.visible
    }
    FormCard.FormCard {
        id: localizationCard
        visible: forceMetricSwitch.visible

        FormCard.FormSwitchDelegate {
            id: forceMetricSwitch
            text: i18n("Use metric units")
            description: i18n("Use metric units for distance and temperature values regardless of the platform region settings.")
            checked: Settings.forceMetricUnits
            onToggled: Settings.forceMetricUnits = checked
            visible: Qt.locale().measurementSystem !== Locale.MetricSystem || Settings.homeCountryIsoCode === "US"
        }
    }

    // Online services
    FormCard.FormHeader {
        title: i18n("Online Services")
    }
    FormCard.FormCard {
        FormCard.FormCheckDelegate {
            text: i18n("Query Traffic Data")
            checked: Settings.queryLiveData
            onToggled: Settings.queryLiveData = checked
            description: i18n("When enabled, this will query transport provider online services for changes such as delays or gate and platform changes.")
        }

        FormCard.FormDelegateSeparator {}

        FormCard.FormCheckDelegate {
            text: i18n("Use insecure services")
            checked: LiveDataManager.publicTransportManager.allowInsecureBackends
            onToggled: LiveDataManager.publicTransportManager.allowInsecureBackends = checked
            description: i18n("Enabling this will also use online services that do not offer transport encryption. This is not recommended, but might be unavoidable when relying on live data from certain providers.")
            descriptionItem.color: Kirigami.Theme.negativeTextColor
        }

        FormCard.FormDelegateSeparator {}

        FormCard.FormButtonDelegate {
            Kirigami.FormData.isSection: true
            text: i18n("Public Transport Information Sources…")
            icon.name: "settings-configure"
            onClicked: applicationWindow().pageStack.layers.push(ptBackendPage)
        }

        FormCard.FormDelegateSeparator {}

        FormCard.FormCheckDelegate {
            id: weatherSwitch
            text: i18n("Weather Forecast")
            checked: Settings.weatherForecastEnabled
            onToggled: Settings.weatherForecastEnabled = checked
            description: i18n("Showing weather forecasts will query online services.")
        }

        // ATTENTION do not remove this note, see https://api.met.no/license_data.html
        FormCard.FormTextDelegate {
            description: i18n("Using data from <a href=\"https://www.met.no/\">The Norwegian Meteorological Institute</a> under <a href=\"https://creativecommons.org/licenses/by/4.0\">Creative Commons 4.0 BY International</a> license.")
            visible: weatherSwitch.checked
            onLinkActivated: Qt.openUrlExternally(link)
        }

        FormCard.FormDelegateSeparator {}

        FormCard.FormCheckDelegate {
            id: autoMapDownload
            text: i18n("Preload Map Data")
            checked: Settings.preloadMapData
            onToggled: Settings.preloadMapData = checked
            description: i18n("Enabling this will download maps for all stations and airports for upcoming trips when connected to Wifi network.")
        }

        FormCard.FormDelegateSeparator {}

        FormCard.FormCheckDelegate {
            id: currencyConversion
            text: i18n("Currency Conversion")
            checked: Settings.performCurrencyConversion
            onToggled: {
                Settings.performCurrencyConversion = checked
                if (checked) {
                    UnitConversion.syncCurrencyConversionTable();
                }
            }
            description: i18n("Enabling this will perform online queries for exchange rates to currencies at travel destinations.")
        }

        FormCard.FormDelegateSeparator {}

        FormCard.FormCheckDelegate {
            text: i18n("Wikimedia Online Content")
            checked: Settings.wikimediaOnlineContentEnabled
            onToggled: Settings.wikimediaOnlineContentEnabled = checked
            description: i18n("Use additional online content from Wikimedia and Wikidata.")
        }
    }

    // Transfer assistant
    FormCard.FormHeader {
        Kirigami.FormData.isSection: true
        title: i18n("Transfer Assistant")
    }
    FormCard.FormCard {
        FormCard.FormCheckDelegate {
            text: i18n("Automatically add transfers")
            checked: Settings.autoAddTransfers
            onToggled: Settings.autoAddTransfers = checked
        }

        FormCard.FormDelegateSeparator {}

        FormCard.FormCheckDelegate {
            text: i18n("Automatically fill transfers")
            description: i18n("When enabled, this will query transport provider online services automatically for transfer information.")
            checked: Settings.autoFillTransfers
            onToggled: Settings.autoFillTransfers = checked
            enabled: Settings.autoAddTransfers && Settings.queryLiveData
        }
    }


    // Notifications
    FormCard.FormHeader {
        Kirigami.FormData.isSection: true
        title: i18n("Notifications")
    }
    FormCard.FormCard {
        id: notificationCard
        FormCard.FormButtonDelegate {
            text: i18n("Configure Notifications…")
            icon.name: "notifications"
            onClicked: NotificationConfigController.configureNotifications()
            enabled: NotificationConfigController.canConfigureNotification
        }

        FormCard.FormDelegateSeparator {}

        FormCard.FormCheckDelegate {
            text: i18n("Show notifications on lock screen")
            checked: Settings.showNotificationOnLockScreen
            onToggled: Settings.showNotificationOnLockScreen = checked
            enabled: NotificationConfigController.canShowOnLockScreen
        }

        FormCard.FormDelegateSeparator {}

        property bool hasNotificationPermission: KNotification.NotificationPermission.checkPermission()
        FormCard.FormButtonDelegate {
            text: i18n("Request permissions…")
            description: i18n("Additional permissions are required to show notifications.")
            icon.name: "documentinfo"
            icon.color: Kirigami.Theme.neutralTextColor
            visible: !notificationCard.hasNotificationPermission
            function permissionCallback(success) {
                notificationCard.hasNotificationPermission = success;
            }
            onClicked: KNotification.NotificationPermission.requestPermission(permissionCallback)
        }
        FormCard.FormTextDelegate {
            text: i18n("Notification permissions are available")
            description: i18n("No further action required.");
            icon.name: "checkmark"
            icon.color: Kirigami.Theme.positiveTextColor
            visible: notificationCard.hasNotificationPermission
        }
    }

    // Wifi access for onboard status information
    property OnboardStatus onboardStatus: OnboardStatus {
        id: onboardStatus
    }
    FormCard.FormHeader {
        Kirigami.FormData.isSection: true
        title: i18n("Onboard Status")
    }
    FormCard.FormCard {
        FormCard.FormButtonDelegate {
            text: i18n("Request permissions…")
            description: i18n("Additional permissions are required to access the Wi-Fi status.")
            icon.name: "documentinfo"
            icon.color: Kirigami.Theme.neutralTextColor
            visible: onboardStatus.status == OnboardStatus.MissingPermissions
            onClicked: onboardStatus.requestPermissions()
        }
        FormCard.FormTextDelegate {
            text: {
                switch (onboardStatus.status) {
                    case OnboardStatus.NotConnected:
                    case OnboardStatus.Onboard:
                        return i18n("Wi-Fi access for onboard information is available.");
                    case OnboardStatus.WifiNotEnabled:
                        return i18n("Wi-Fi is not enabled");
                    case OnboardStatus.LocationServiceNotEnabled:
                        return i18n("Location service is not enabled");
                    case OnboardStatus.NotAvailable:
                        return i18n("Wi-Fi access is not available on this system");
                }
            }
            description: {
                switch (onboardStatus.status) {
                    case OnboardStatus.NotConnected:
                    case OnboardStatus.Onboard:
                        return i18n("No further action required.");
                    case OnboardStatus.WifiNotEnabled:
                        return i18n("Enable Wi-Fi on your system to access onboard information.");
                    case OnboardStatus.LocationServiceNotEnabled:
                        return i18n("Enable the location service on your device to access onboard information.");
                    case OnboardStatus.NotAvailable:
                        return i18n("Unfortunately, retrieving onboard information is not supported on your device at this time.");
                }
            }
            icon.name: {
                switch (onboardStatus.status) {
                    case OnboardStatus.NotConnected:
                    case OnboardStatus.Onboard:
                        return "checkmark"
                    case OnboardStatus.WifiNotEnabled:
                    case OnboardStatus.LocationServiceNotEnabled:
                        return "documentinfo"
                    case OnboardStatus.NotAvailable:
                        return "dialog-cancel"
                }
            }
            icon.color: {
                switch (onboardStatus.status) {
                    case OnboardStatus.NotConnected:
                    case OnboardStatus.Onboard:
                        return Kirigami.Theme.positiveTextColor
                    case OnboardStatus.WifiNotEnabled:
                    case OnboardStatus.LocationServiceNotEnabled:
                        return Kirigami.Theme.neutralTextColor
                    case OnboardStatus.NotAvailable:
                        return Kirigami.Theme.negativeTextColor
                }
            }
            enabled: onboardStatus.status != OnboardStatus.MissingPermissions
        }
    }

    FormCard.FormHeader {
        title: i18n("Matrix Integration")
        visible: matrixCard.visible
    }
    FormCard.FormCard {
        id: matrixCard
        visible: MatrixController.isAvailable
        FormCard.FormTextDelegate {
            text: MatrixController.manager.infoString.length > 0 ? MatrixController.manager.infoString : MatrixController.manager.connected ? i18n("Logged in as %1", MatrixController.manager.userId) : ""
            visible: text.length > 0
        }
        FormCard.FormButtonDelegate {
            text: i18n("Synchronize rooms")
            icon.name: "view-refresh"
            onClicked: MatrixController.manager.sync()
            visible: MatrixController.manager.connected
        }
        FormCard.FormTextFieldDelegate {
            id: matrixId
            label: i18n("Matrix ID")
            visible: !MatrixController.manager.connected
            text: MatrixController.manager.userId
        }
        FormCard.FormDelegateSeparator {
            visible: matrixId.visible
        }
        FormCard.FormTextFieldDelegate {
            id: matrixPassword
            label: i18n("Password")
            echoMode: TextInput.Password
            visible: !MatrixController.manager.connected
            onAccepted: {
                MatrixController.manager.login(matrixId.text, matrixPassword.text);
            }
        }
        FormCard.FormDelegateSeparator {
            visible: matrixSessionVerified.visible
        }
        FormCard.FormTextDelegate {
            id: matrixSessionVerified
            text: MatrixController.manager.isVerifiedSession ? i18nc("Matrix session verification", "Session verified") : i18nc("Matrix session verification", "Session not verified.")
            description: MatrixController.manager.isVerifiedSession ? i18n("No further action required.") : i18nc("Matrix session verification", "Initiate session verification in your Matrix chat client.")
            icon.name: MatrixController.manager.isVerifiedSession ? "security-high" : "security-low"
            visible: MatrixController.manager.connected
        }
        FormCard.FormDelegateSeparator { above: loginButton }
        FormCard.FormButtonDelegate {
            id: loginButton
            text: MatrixController.manager.connected ? i18n("Log out") : i18n("Log in")
            onClicked: MatrixController.manager.connected ? MatrixController.manager.logout() : MatrixController.manager.login(matrixId.text, matrixPassword.text)
            enabled: MatrixController.manager.connected || (matrixId.text.length > 0 && matrixPassword.text.length > 0)
        }
    }

    FormCard.FormHeader {
        title: i18n("Träwelling Integration")
    }

    FormCard.FormCard {
        FormCard.FormTextDelegate {
            description: i18nc("@info", "Träwelling is a free check-in service that lets you tell your friends where you are and where you can log your public transit journeys. For more information, visit <a href=\"https://traewelling.de\">traewelling.de</a>");
            onLinkActivated: link => Qt.openUrlExternally(link)
        }
        FormCard.FormTextDelegate {
            visible: TraewellingController.isLoggedIn
            text: i18n("Logged in as %1", TraewellingController.username)
        }
        FormCard.FormButtonDelegate {
            text: i18n("Login")
            onClicked: TraewellingController.login()
            visible: !TraewellingController.isLoggedIn
        }
        FormCard.FormButtonDelegate {
            text: i18n("Logout")
            onClicked: TraewellingController.logout()
            visible: TraewellingController.isLoggedIn
        }
    }

    FormCard.FormHeader {
        title: i18n("Contributing")
    }
    FormCard.FormCard {
        FormCard.FormCheckDelegate {
            text: i18n("I contribute to OpenStreetMap")
            description: i18n("Enables OSM editing options.")
            checked: Settings.osmContributorMode
            onToggled: Settings.osmContributorMode = checked
        }
    }
}
