// SPDX-FileCopyrightText: 2018 Volker Krause <vkrause@kde.org>
// SPDX-FileCopyrightText: 2022 Carl Schwan <carl@carlschwan.eu>
// SPDX-License-Identifier: LGPL-2.0-or-later

import QtQuick
import QtQuick.Layouts
import QtQuick.Controls as QQC2
import org.kde.kirigami as Kirigami
import org.kde.kirigamiaddons.formcard as FormCard
import org.kde.kirigamiaddons.components as KirigamiComponents
import org.kde.kpublictransport as KPublicTransport
import org.kde.kpublictransport.ui as KPublicTransport
import org.kde.kitinerary
import org.kde.itinerary

DetailsPage {
    id: root

    title: i18n("Train Ticket")
    editor: TrainEditor {
        controller: root.controller
    }

    data: BarcodeScanModeButton {
        id: scanModeButton
        page: root
        visible: ticketToken.hasBarcode
    }

    Component {
        id: alternativePage
        AlternativeJourneyPage {
            controller: root.controller
            publicTransportManager: LiveDataManager.publicTransportManager
        }
    }

    Component {
        id: vehicleLayoutPage
        VehicleLayoutPage {
            publicTransportManager: root.controller.liveDataManager.publicTransportManager
            selectedVehicleSection: root.reservation.reservedTicket.ticketedSeat.seatSection
            selectedClasses: root.reservation.reservedTicket.ticketedSeat.seatingType
            seat: root.reservation.reservedTicket.ticketedSeat.seatNumber

            property bool arrival

            onLayoutUpdated: root.controller.setVehicleLayout(vehicleLayout, arrival);
        }
    }

    header: KirigamiComponents.Banner {
        id: banner

        showCloseButton: true
        visible: false
    }

    ColumnLayout {
        spacing: 0

        KPublicTransport.TransportIcon {
            id: transportIcon
            Layout.alignment: Qt.AlignHCenter
            Layout.fillWidth: true
            // A bit of extra spacing since the logos often have no padding.
            Layout.bottomMargin: root.departure.route.line.hasLogo || root.departure.route.line.hasModeLogo ? Kirigami.Units.largeSpacing : 0
            iconHeight: Kirigami.Units.iconSizes.medium
            source: root.departure.route.line.mode === KPublicTransport.Line.Unknown ? ReservationHelper.defaultIconName(root.reservation) : root.departure.route.line.iconName
            enabled: !root.controller.isCanceled
        }

        Kirigami.Heading {
            text: root.reservationFor.trainName + " " + root.reservationFor.trainNumber
            wrapMode: Text.WordWrap
            horizontalAlignment: Text.AlignHCenter
            textFormat: Text.PlainText
            visible: !root.departure.route.line.hasLogo && (root.reservationFor.trainName || root.reservationFor.trainNumber)
            enabled: !root.controller.isCanceled

            leftPadding: Kirigami.Units.smallSpacing
            rightPadding: Kirigami.Units.smallSpacing
            bottomPadding: Kirigami.Units.smallSpacing

            Layout.fillWidth: true
        }

        Kirigami.Heading {
            text: i18n("%1 to %2", root.reservationFor.departureStation.name, root.reservationFor.arrivalStation.name)

            level: 2
            wrapMode: Text.WordWrap
            horizontalAlignment: Text.AlignHCenter
            textFormat: Text.PlainText
            enabled: !root.controller.isCanceled

            leftPadding: Kirigami.Units.smallSpacing
            rightPadding: Kirigami.Units.smallSpacing
            bottomPadding: Kirigami.Units.smallSpacing

            Layout.fillWidth: true
        }

        FormCard.FormCard {
            visible: ticketToken.hasContent
            // ticket barcode
            TicketTokenDelegate {
                id: ticketToken
                Layout.fillWidth: true
                batchId: root.batchId
                onCurrentReservationIdChanged: {
                    if (!currentReservationId)
                        return;
                    root.currentReservationId = currentReservationId;
                }
                onScanModeToggled: scanModeController.toggle()
            }
        }

        // departure data
        FormCard.FormHeader {
            title: i18nc("Train departure", "Departure")
        }

        FormCard.FormCard {
            FormCard.FormTextDelegate {
                id: departureTimeDelegate
                text: i18n("Departure time")
                visible: departureTimeLabel.text.length > 0
                contentItem: ColumnLayout {
                    spacing: Kirigami.Units.smallSpacing
                    QQC2.Label {
                        text: departureTimeDelegate.text
                        Layout.fillWidth: true
                        elide: Text.ElideRight
                        Accessible.ignored: true
                    }
                    RowLayout {
                        Layout.fillWidth: true
                        QQC2.Label {
                            id: departureTimeLabel
                            text: Localizer.formatDateTime(reservationFor, "departureTime")
                            color: Kirigami.Theme.disabledTextColor
                            font: Kirigami.Theme.smallFont
                            elide: Text.ElideRight
                        }
                        QQC2.Label {
                            text: (departure.departureDelay >= 0 ? "+" : "") + departure.departureDelay
                            color: (departure.departureDelay > 1) ? Kirigami.Theme.negativeTextColor : Kirigami.Theme.positiveTextColor
                            visible: departure.hasExpectedDepartureTime
                            Accessible.ignored: !visible
                        }
                    }
                }
            }
            FormCard.FormTextDelegate {
                text: i18n("Departure date")
                visible: !departureTimeDelegate.visible && text.length > 0
                description: Localizer.formatDate(reservationFor, "departureDay")
            }

            FormCard.FormDelegateSeparator { visible: departureStation.visible }

            FormPlaceDelegate {
                id: departureStation

                text: i18nc("train station", "Station")
                placeName: reservationFor.departureStation.name
                place: reservationFor.departureStation
                controller: root.controller
                isRangeBegin: true
                visible: description
            }

            FormCard.FormDelegateSeparator { visible: departurePlatformDelegate.visible }

            FormPlatformDelegate {
                id: departurePlatformDelegate
                text: i18nc("train station platform", "Platform")
                stopover: root.departure
                sections: root.controller.departurePlatformSections
                scheduledPlatform: root.reservationFor.departurePlatform
            }

            FormCard.FormDelegateSeparator { visible: departureNotes.visible }
            FormCard.FormTextDelegate {
                id: departureNotes
                text: i18n("Additional information")
                description: root.controller.journey.notes.concat(root.departure.notes).join("<br/>")
                descriptionItem.textFormat: Text.RichText
                descriptionItem.wrapMode: Text.Wrap
                visible: description !== ""
                font.italic: true
                onLinkActivated: (link) => { Qt.openUrlExternally(link); }
            }

            FormCard.FormDelegateSeparator { visible: departureLayoutButton.visible }

            FormCard.FormButtonDelegate {
                id: departureLayoutButton
                text: i18n("Departure Vehicle Layout")
                icon.name: "view-list-details"
                visible: departure && (departure.route.line.mode == KPublicTransport.Line.LongDistanceTrain || departure.route.line.mode == KPublicTransport.Line.Train || departure.route.name !== "")
                onClicked: applicationWindow().pageStack.push(vehicleLayoutPage, {stopover: root.controller.departure, arrival: false})
            }
        }

        // arrival data
        FormCard.FormHeader {
            title: i18nc("Train arrival", "Arrival")
        }

        FormCard.FormCard {
            FormCard.FormTextDelegate {
                id: arrivalTimeDelegate
                text: i18n("Arrival time")
                visible: arrivalTimeLabel.text.length > 0
                contentItem: ColumnLayout {
                    Layout.fillWidth: true
                    spacing: Kirigami.Units.smallSpacing
                    QQC2.Label {
                        text: arrivalTimeDelegate.text
                        Layout.fillWidth: true
                        elide: Text.ElideRight
                        Accessible.ignored: true
                    }
                    RowLayout {
                        Layout.fillWidth: true
                        QQC2.Label {
                            id: arrivalTimeLabel
                            text: Localizer.formatDateTime(reservationFor, "arrivalTime")
                            color: Kirigami.Theme.disabledTextColor
                            font: Kirigami.Theme.smallFont
                            elide: Text.ElideRight
                        }
                        QQC2.Label {
                            font: Kirigami.Theme.smallFont
                            text: (arrival.arrivalDelay >= 0 ? "+" : "") + arrival.arrivalDelay
                            color: (arrival.arrivalDelay > 1) ? Kirigami.Theme.negativeTextColor : Kirigami.Theme.positiveTextColor
                            visible: arrival.hasExpectedArrivalTime
                            Accessible.ignored: !visible
                        }
                    }
                }
            }

            FormCard.FormDelegateSeparator { visible: arrivalTimeLabel.text.length > 0 && arrivalStation.visible }

            FormPlaceDelegate {
                id: arrivalStation

                text: i18nc("train station", "Station")
                description: reservationFor.arrivalStation.name
                place: reservationFor.arrivalStation
                controller: root.controller
                isRangeEnd: true
                visible: description
            }

            FormCard.FormDelegateSeparator { visible: arrivalPlatformDelegate.visible }

            FormPlatformDelegate {
                id: arrivalPlatformDelegate
                text: i18nc("train station platform", "Platform")
                stopover: root.arrival
                sections: root.controller.arrivalPlatformSections
                scheduledPlatform: root.reservationFor.arrivalPlatform
            }

            FormCard.FormDelegateSeparator { visible: arrivalNotes.visible }
            FormCard.FormTextDelegate {
                id: arrivalNotes
                text: i18n("Additional information")
                description: root.arrival.notes.join("<br/>")
                descriptionItem.textFormat: Text.RichText
                descriptionItem.wrapMode: Text.Wrap
                visible: description !== ""
                font.italic: true
                onLinkActivated: (link) => { Qt.openUrlExternally(link); }
            }

            FormCard.FormDelegateSeparator { visible: arrivalLayoutButton.visible }

            FormCard.FormButtonDelegate {
                id: arrivalLayoutButton
                text: i18n("Arrival Vehicle Layout")
                icon.name: "view-list-details"
                visible: arrival && (arrival.route.line.mode == KPublicTransport.Line.LongDistanceTrain || arrival.route.line.mode == KPublicTransport.Line.Train || arrival.route.name !== "")
                onClicked: applicationWindow().pageStack.push(vehicleLayoutPage, {stopover: root.controller.arrival, arrival: true});
            }
        }

        // seat reservation
        FormCard.FormHeader {
            visible: seatCard.visible
            title: i18nc("Train seat", "Seat")
        }

        SeatRowFormCard {
            id: seatCard

            visible: root.hasSeat

            TimelineDelegateSeatRowLabel {
                text: i18nc("train coach", "Coach: <b>%1</b>", root.seatSectionString)
            }

            Kirigami.Separator {
                Layout.fillHeight: true
            }
            TimelineDelegateSeatRowLabel {
                text: i18nc("train seat", "Seat: <b>%1</b>", root.seatString)
            }
            Kirigami.Separator {
                Layout.fillHeight: true
            }
            TimelineDelegateSeatRowLabel {
                text: {
                    const s = root.reservation?.reservedTicket?.ticketedSeat?.seatingType;
                    return i18nc("train class", "Class: <b>%1</b>", s || "-");
                }
                lowPriority: true
            }
        }

        ProgramMembershipCard {
            programMembership: root.reservation.programMembershipUsed
        }

        BookingCard {
            reservation: root.reservation
        }

        ReservationDocumentsCard {
            controller: root.controller
        }

        ActionsCard {
            batchId: root.batchId
            reservationId: root.currentReservationId
            editor: root.editor
            reservation: root.reservation
            additionalActions: [
                QQC2.Action {
                    enabled: TraewellingController.isLoggedIn
                    text: i18nc("@action:button", "Add to Träwelling")
                    icon.name: "cloud-upload"
                    onTriggered: {
                        banner.visible = false;
                        TraewellingController.checkin(reservationFor.departureStation.name, reservationFor.arrivalStation.name, reservationFor.departureTime, reservationFor.arrivalTime, departure.route.direction);
                    }
                },
                QQC2.Action {
                    text: i18n("Alternatives")
                    icon.name: "clock"
                    onTriggered: applicationWindow().pageStack.push(alternativePage)
                    enabled: reservationFor.departureStation.geo.isValid && reservationFor.arrivalStation.geo.isValid
                },
                Kirigami.Action {
                    text: i18n("Journey Details")
                    icon.name: "view-calendar-day"
                    onTriggered: applicationWindow().pageStack.push(journeySectionPage, {
                        journeySection: root.controller.trip,
                        departureStopIndex: root.controller.tripDepartureIndex,
                        arrivalStopIndex: root.controller.tripArrivalIndex,
                        showProgress: root.controller.isCurrent
                    });
                    Component.onCompleted: {
                        visible = Qt.binding(function() { return root.controller.journey && (root.controller.journey.intermediateStops.length > 0 || !root.controller.journey.path.isEmpty); });
                    }
                },
                Kirigami.Action {
                    text: "Test Notification"
                    icon.name: "notifications"
                    visible: Settings.developmentMode
                    onTriggered: LiveDataManager.showNotification(root.batchId)
                }
            ]
        }

        // spacer for the floating buttons
        Item {
            visible: scanModeButton.visible
            implicitHeight: root.width < Kirigami.Units.gridUnit * 30 + scanModeButton.width * 2 ? scanModeButton.height : 0
        }

        Connections {
            target: TraewellingController
            function onUploadStatus(status): void {
                banner.visible = true;
                banner.text = status === TraewellingController.Success ? i18n("Added to Träwelling") : i18n("Failed to add to Träwelling");
                banner.type = status === TraewellingController.Success ? Kirigami.MessageType.Positive : Kirigami.MessageType.Error;
            }
        }
    }
}
