// SPDX-FileCopyrightText: 2025 Pedro Hernandez <pehg_@outlook.com>
//
// SPDX-License-Identifier: GPL-3.0-or-later

#include <QTest>
// Kaidan
#include "RosterItem.h"
#include "Test.h"
#include "TestUtils.h"

class RosterItemTest : public Test
{
    Q_OBJECT

private Q_SLOTS:
    void comparison_data()
    {
        QTest::addColumn<RosterItem>("item1");
        QTest::addColumn<RosterItem>("item2");
        QTest::addColumn<bool>("lessThan");
        QTest::addColumn<bool>("lessThanOrEqual");
        QTest::addColumn<bool>("equal");
        QTest::addColumn<bool>("greaterThanOrEqual");
        QTest::addColumn<bool>("greaterThan");

        RosterItem baseItem;
        baseItem.accountJid = QStringLiteral("alice@example.org");
        baseItem.jid = QStringLiteral("bob@example.com");
        baseItem.name = QStringLiteral("Bob");
        baseItem.lastMessageDateTime = QDateTime::fromString(QStringLiteral("2025-01-01T00:00:00"), Qt::ISODate);

        RosterItem copiedtem{baseItem};

        QTest::newRow("copy") << baseItem << copiedtem << false << true << true << true << false;

        auto pinnedItem = baseItem;
        pinnedItem.pinningPosition = 1;
        QTest::newRow("pinningPosition") << baseItem << pinnedItem << false << false << false << true << true;

        auto pinnedLowerDraftItem = baseItem;
        pinnedLowerDraftItem.pinningPosition = 1;
        pinnedLowerDraftItem.lastMessageDeliveryState = Enums::DeliveryState::Draft;

        auto pinnedHigherDraftItem = baseItem;
        pinnedHigherDraftItem.pinningPosition = 2;
        pinnedHigherDraftItem.lastMessageDeliveryState = Enums::DeliveryState::Draft;
        QTest::newRow("both pinned, both draft") << pinnedLowerDraftItem << pinnedHigherDraftItem << false << false << false << true << true;

        auto pinnedHigherSentItem = baseItem;
        pinnedHigherSentItem.pinningPosition = 2;
        pinnedHigherSentItem.lastMessageDeliveryState = Enums::DeliveryState::Sent;
        QTest::newRow("both pinned, item1 draft") << pinnedLowerDraftItem << pinnedHigherSentItem << true << true << false << false << false;

        auto unpinnedDraftItem = baseItem;
        unpinnedDraftItem.lastMessageDeliveryState = Enums::DeliveryState::Draft;

        auto pinnedSentItem = baseItem;
        pinnedSentItem.pinningPosition = 1;
        pinnedSentItem.lastMessageDeliveryState = Enums::DeliveryState::Sent;
        QTest::newRow("item1 draft but unpinned, item2 no draft but pinned") << unpinnedDraftItem << pinnedSentItem << false << false << false << true << true;

        auto unpinnedSentItem = baseItem;
        unpinnedSentItem.lastMessageDeliveryState = Enums::DeliveryState::Sent;
        QTest::newRow("both unpinned, item1 draft") << unpinnedDraftItem << unpinnedSentItem << true << true << false << false << false;

        auto moreRecentTimeItem = baseItem;
        moreRecentTimeItem.lastMessageDateTime = QDateTime::fromString(QStringLiteral("2025-01-01T00:00:01"), Qt::ISODate);
        QTest::newRow("lastMessageDateTime") << baseItem << moreRecentTimeItem << false << false << false << true << true;

        auto differentNameItem = baseItem;
        differentNameItem.name = QStringLiteral("Carol");
        QTest::newRow("name") << baseItem << differentNameItem << true << true << false << false << false;

        auto differentAccountItem = baseItem;
        differentAccountItem.accountJid = QStringLiteral("dave@example.net");
        QTest::newRow("accountJid") << baseItem << differentAccountItem << true << true << false << false << false;
    }

    void comparison()
    {
        QFETCH(RosterItem, item1);
        QFETCH(RosterItem, item2);
        QFETCH(bool, lessThan);
        QFETCH(bool, greaterThan);
        QFETCH(bool, equal);
        QFETCH(bool, lessThanOrEqual);
        QFETCH(bool, greaterThanOrEqual);

        // Test the comparison operators generated by the spaceship operator.
        QCOMPARE(item1 < item2, lessThan);
        QCOMPARE(item1 > item2, greaterThan);
        QCOMPARE(item1 == item2, equal);
        QCOMPARE(item1 <= item2, lessThanOrEqual);
        QCOMPARE(item1 >= item2, greaterThanOrEqual);

        // Test the spaceship operator's result directly.
        if (lessThan) {
            QVERIFY((item1 <=> item2) == std::strong_ordering::less);
        } else if (greaterThan) {
            QVERIFY((item1 <=> item2) == std::strong_ordering::greater);
        } else if (equal) {
            QVERIFY((item1 <=> item2) == std::strong_ordering::equal);
        }
    }
};

QTEST_GUILESS_MAIN(RosterItemTest)
#include "RosterItemTest.moc"
