/*
   SPDX-FileCopyrightText: 2021-2026 Laurent Montel <montel@kde.org>

   SPDX-License-Identifier: LGPL-2.0-or-later
*/

#include "confirmbeforedeletingrule.h"
using namespace Qt::Literals::StringLiterals;

#include "confirmbeforedeletingplugin_debug.h"
#include <Akonadi/MessageStatus>
#include <KLocalizedString>

#include <QDebug>

ConfirmBeforeDeletingRule::ConfirmBeforeDeletingRule() = default;

ConfirmBeforeDeletingRule::~ConfirmBeforeDeletingRule() = default;

const QString &ConfirmBeforeDeletingRule::pattern() const
{
    return mPattern;
}

void ConfirmBeforeDeletingRule::setPattern(const QString &newPattern)
{
    mPattern = newPattern;
}

ConfirmBeforeDeletingRule::RuleType ConfirmBeforeDeletingRule::ruleType() const
{
    return mRuleType;
}

void ConfirmBeforeDeletingRule::setRuleType(RuleType newRuleType)
{
    mRuleType = newRuleType;
}

void ConfirmBeforeDeletingRule::load(const KConfigGroup &group)
{
    const QString patternValue = group.readEntry(u"Pattern"_s, QString());
    const QString checkType = group.readEntry(u"Type"_s, QString());
    setPattern(patternValue);
    setRuleType(stringToRuleType(checkType));
}

bool ConfirmBeforeDeletingRule::isValid() const
{
    return (!mPattern.isEmpty() && (mRuleType != Unknown)) || (mRuleType == Unread) || (mRuleType == Important);
}

void ConfirmBeforeDeletingRule::save(KConfigGroup &group) const
{
    group.writeEntry(u"Pattern"_s, mPattern);
    group.writeEntry(u"Type"_s, ruleTypeToString(mRuleType));
}

ConfirmBeforeDeletingRule::RuleType ConfirmBeforeDeletingRule::stringToRuleType(const QString &str)
{
    if (str == QLatin1StringView("body")) {
        return Body;
    } else if (str == QLatin1StringView("subject")) {
        return Subject;
    } else if (str == QLatin1StringView("to")) {
        return To;
    } else if (str == QLatin1StringView("cc")) {
        return Cc;
    } else if (str == QLatin1StringView("bcc")) {
        return Bcc;
    } else if (str == QLatin1StringView("unread")) {
        return Unread;
    } else if (str == QLatin1StringView("important")) {
        return Important;
    }
    return Unknown;
}

bool ConfirmBeforeDeletingRule::operator==(const ConfirmBeforeDeletingRule &other) const
{
    return mPattern == other.pattern() && mRuleType == other.ruleType();
}

QString ConfirmBeforeDeletingRule::ruleTypeToString(ConfirmBeforeDeletingRule::RuleType r)
{
    QString tmp;
    switch (r) {
    case Unknown:
        break;
    case Body:
        tmp = u"body"_s;
        break;
    case Subject:
        tmp = u"subject"_s;
        break;
    case To:
        tmp = u"to"_s;
        break;
    case Cc:
        tmp = u"cc"_s;
        break;
    case Bcc:
        tmp = u"bcc"_s;
        break;
    case Unread:
        tmp = u"unread"_s;
        break;
    case Important:
        tmp = u"important"_s;
        break;
    }
    return tmp;
}

bool ConfirmBeforeDeletingRule::deletingNeedToConfirm(const Akonadi::Item &item, QString &checkFoundInfo) const
{
    bool needToConfirm = false;
    if (item.hasPayload<std::shared_ptr<KMime::Message>>()) {
        auto msg = item.payload<std::shared_ptr<KMime::Message>>();
        switch (mRuleType) {
        case Body: {
            const auto ba = msg->body();
            if (QString::fromUtf8(ba).contains(pattern())) {
                needToConfirm = true;
                generateConfirmMessageInfo(msg, checkFoundInfo);
            }
            break;
        }
        case Subject: {
            if (auto subject = msg->subject(KMime::CreatePolicy::DontCreate)) {
                const QString subjectStr = subject->asUnicodeString();
                if (subjectStr.contains(pattern())) {
                    needToConfirm = true;
                    generateConfirmMessageInfo(msg, checkFoundInfo);
                }
            }
            break;
        }
        case To: {
            if (auto to = msg->to(KMime::CreatePolicy::DontCreate)) {
                const QString toStr = to->asUnicodeString();
                if (toStr.contains(pattern())) {
                    needToConfirm = true;
                    generateConfirmMessageInfo(msg, checkFoundInfo);
                }
            }
            break;
        }
        case Cc: {
            if (auto cc = msg->cc(KMime::CreatePolicy::DontCreate)) {
                const QString ccStr = cc->asUnicodeString();
                if (ccStr.contains(pattern())) {
                    needToConfirm = true;
                    generateConfirmMessageInfo(msg, checkFoundInfo);
                }
            }
            break;
        }
        case Bcc: {
            if (auto bcc = msg->bcc(KMime::CreatePolicy::DontCreate)) {
                const QString bccStr = bcc->asUnicodeString();
                if (bccStr.contains(pattern())) {
                    needToConfirm = true;
                    generateConfirmMessageInfo(msg, checkFoundInfo);
                }
            }
            break;
        }
        case Unread: {
            Akonadi::MessageStatus status;
            status.setStatusFromFlags(item.flags());
            if (!status.isRead()) {
                needToConfirm = true;
                generateConfirmMessageInfoFromStatus(msg, checkFoundInfo, i18n("Unread"));
            }
            break;
        }
        case Important: {
            Akonadi::MessageStatus status;
            status.setStatusFromFlags(item.flags());
            if (status.isImportant()) {
                needToConfirm = true;
                generateConfirmMessageInfoFromStatus(msg, checkFoundInfo, i18n("Important"));
            }
            break;
        }

        case Unknown:
            qCWarning(KMAIL_CONFIRMBEFOREDELETING_PLUGIN_LOG) << "Invalid rules!";
            break;
        }
    }

    return needToConfirm;
}

void ConfirmBeforeDeletingRule::generateConfirmMessageInfo(const std::shared_ptr<KMime::Message> &msg, QString &checkFoundInfo) const
{
    QString subjectStr;
    if (auto subject = msg->subject(KMime::CreatePolicy::DontCreate)) {
        subjectStr = subject->asUnicodeString();
    }
    checkFoundInfo = i18n("The message with subject \'%2\' contains \'%1\'", pattern(), subjectStr);
}

void ConfirmBeforeDeletingRule::generateConfirmMessageInfoFromStatus(const std::shared_ptr<KMime::Message> &msg,
                                                                     QString &checkFoundInfo,
                                                                     const QString &statusStr) const
{
    QString subjectStr;
    if (auto subject = msg->subject(KMime::CreatePolicy::DontCreate)) {
        subjectStr = subject->asUnicodeString();
    }
    checkFoundInfo = i18n("The message with subject \'%1\' is %2", subjectStr, statusStr);
}

QDebug operator<<(QDebug d, const ConfirmBeforeDeletingRule &t)
{
    d << "Pattern " << t.pattern();
    d << "mRuleType " << t.ruleType();
    return d;
}

#include "moc_confirmbeforedeletingrule.cpp"
