/*
 * SPDX-FileCopyrightText: (C) 2015 Vishesh Handa <vhanda@kde.org>
 * SPDX-FileCopyrightText: (C) 2017 Atul Sharma <atulsharma406@gmail.com>
 * SPDX-FileCopyrightText: (C) 2017 Marco Martin <mart@kde.org>
 * SPDX-FileCopyrightText: (C) 2021 Noah Davis <noahadvs@gmail.com>
 * SPDX-FileCopyrightText: (C) 2021 Mikel Johnson <mikel5764@gmail.com>
 * SPDX-FileCopyrightText: (C) 2021 Arjen Hiemstra <ahiemstra@heimr.nl>
 *
 * SPDX-License-Identifier: LGPL-2.1-only OR LGPL-3.0-only OR LicenseRef-KDE-Accepted-LGPL
 */

import QtQuick
import QtQuick.Controls as Controls
import QtQml
import org.kde.kirigami as Kirigami
import org.kde.koko as Koko

import ".."

ZoomArea {
    id: root

    required property url source

    required property bool loaded
    required property bool loading

    required property real sourceWidth
    required property real sourceHeight

    required property bool isCurrent

    required property Kirigami.ApplicationWindow mainWindow

    property bool preferAsync: false

    DevicePixelRatioHelper {
        id: dprHelper
        window: root.Window.window
    }

    // Account for scaling: e.g. at a scale of 2x, the image is shown pixel-to-pixel at half size
    implicitContentWidth: sourceWidth / dprHelper.devicePixelRatio
    implicitContentHeight: sourceHeight / dprHelper.devicePixelRatio

    /* Using very small min sizes and very large max sizes since there don't seem
     * to be many good reasons to use more limited ranges.
     *
     * The best tools are simple, but allow users to do complex things without
     * needing to add more complexity.
     * Maybe an artist wants to view the pixels of an image up close to see the
     * exact colors better or shrink an image to see the average colors.
     * We could require the artist to use something like ImageMagick to do that,
     * or we could let them use their favorite image viewer and a color picker to
     * do the same job without having to learn ImageMagick.
     *
     * 8 was picked as the minimum size unless the source size is smaller.
     * It's a fairly arbitrary number. Maybe it could be 1, but that's really
     * really difficult to see and sometimes the single pixel is impossible to see.
     *
     * Media source size times 100 was picked for the max size because that
     * allows Koko to show roughly 19x10 pixels at once when full screen on a
     * 1920x1080 screen at max zoom. An arbitrary number, but it should be fine.
     * QQuickImage is very good at handling large sizes, so unlike Gwenview,
     * performance isn't much of a concern when picking the max size.
     */
    minimumZoomSize: 8
    maximumZoomFactor: 100

    Timer {
        id: doubleClickTimer
        interval: Application.styleHints.mouseDoubleClickInterval + 1
        onTriggered: (root.Controls.ApplicationWindow.window as Koko.Main).controlsVisible = !(root.Controls.ApplicationWindow.window as Koko.Main).controlsVisible
    }

    onClicked: (mouse) => {
        if (mouse.button === Qt.LeftButton) {
            if ((Controls.ApplicationWindow.window as Koko.Main).contextDrawer) {
                (Controls.ApplicationWindow.window as Koko.Main).contextDrawer.drawerOpen = false
            }
            doubleClickTimer.restart()
        }
    }
    onDoubleClicked: (mouse) => {
        if (mouse.button === Qt.LeftButton) {
            doubleClickTimer.stop()
        }
    }

    // TODO: Find a better way to ensure that content size is set to a value
    // without making a binding only when loaded and is the current item.
    // If you set it too early, the zoom will be reset because the delegate may
    // not have the right implicit size yet. An delegate is also not always the
    // current item when loaded.
    property bool isZoomSet: false
    function resetZoomContentSize() {
        if (loaded && isCurrent && !isZoomSet) {
            if (Koko.Config.rememberZoom && Koko.State.zoom * sourceWidth > 1 && Koko.State.zoom * sourceHeight > 1) {
                const size = multiplyContentSize(Koko.State.zoom, implicitContentWidth, implicitContentHeight)
                root.contentX = root.boundedContentX(root.contentX / root.zoomFactor * Koko.State.zoom, size.width)
                root.contentY = root.boundedContentY(root.contentY / root.zoomFactor * Koko.State.zoom, size.height)
                root.contentWidth = size.width
                root.contentHeight = size.height
                isZoomSet = true
            } else {
                root.contentWidth = Qt.binding(() => root.defaultContentRect.width)
                root.contentHeight = Qt.binding(() => root.defaultContentRect.height)
            }
            Koko.State.zoom = root.zoomFactor
        } else {
            isZoomSet = false
        }
    }
    onIsCurrentChanged: resetZoomContentSize()
    onLoadedChanged: resetZoomContentSize()
    onZoomFactorChanged: if (loaded && isCurrent) {
        Koko.State.zoom = root.zoomFactor
    }

    Connections {
        target: Koko.State
        function onZoomChanged() {
            if (!root.loaded || !root.isCurrent || root.dragging || Koko.State.zoom === root.zoomFactor) {
                return
            }
            const size = multiplyContentSize(Koko.State.zoom, implicitContentWidth, implicitContentHeight)
            root.contentX = root.boundedContentX(root.contentX / root.zoomFactor * Koko.State.zoom, size.width)
            root.contentY = root.boundedContentY(root.contentY / root.zoomFactor * Koko.State.zoom, size.height)
            root.contentWidth = size.width
            root.contentHeight = size.height
        }
    }

    Loader {
        anchors.fill: root.contentItem

        active: Koko.Config.imageViewShowCheckerboard
        visible: root.loaded
        sourceComponent: Image {

            readonly property color tileColor: switch (Koko.Config.imageViewBackgroundColor) {
                case 0:
                    return "black";
                case 1:
                    return "white";
                case 2:
                    return Kirigami.Theme.backgroundColor;
            }

            source: "image://checkerboard/" + tileColor
            sourceSize.width: 16
            sourceSize.height: 16

            fillMode: Image.Tile
            smooth: false
            cache: true
        }
    }
}
