/*
   SPDX-FileCopyrightText: 2017-2025 Laurent Montel <montel@kde.org>

   SPDX-License-Identifier: LGPL-2.0-or-later
*/

#include "plugineditorinitmanager.h"
using namespace Qt::Literals::StringLiterals;

#include "messagecomposer_debug.h"
#include "plugineditorinit.h"
#include <KPluginFactory>
#include <KPluginMetaData>
#include <PimCommon/PluginUtil>
#include <QFileInfo>

using namespace MessageComposer;

class PluginEditorInitInfo
{
public:
    PluginEditorInitInfo() = default;

    TextAddonsWidgets::PluginUtilData pluginData;
    QString metaDataFileNameBaseName;
    QString metaDataFileName;
    KPluginMetaData data;
    PluginEditorInit *plugin = nullptr;
    bool isEnabled = true;
};

namespace
{
QString pluginEditorInitVersion()
{
    return u"1.0"_s;
}
}

class MessageComposer::PluginEditorInitManagerPrivate
{
public:
    explicit PluginEditorInitManagerPrivate(PluginEditorInitManager *qq)
        : q(qq)
    {
        initializePlugins();
    }

    void loadPlugin(PluginEditorInitInfo *item);
    [[nodiscard]] QList<PluginEditorInit *> pluginsList() const;
    void initializePlugins();
    [[nodiscard]] QString configPrefixSettingKey() const;
    [[nodiscard]] QString configGroupName() const;
    [[nodiscard]] QList<TextAddonsWidgets::PluginUtilData> pluginsDataList() const;
    [[nodiscard]] PluginEditorInit *pluginFromIdentifier(const QString &id);

    QList<PluginEditorInitInfo> mPluginList;

private:
    QList<TextAddonsWidgets::PluginUtilData> mPluginDataList;
    PluginEditorInitManager *const q;
};

QString PluginEditorInitManagerPrivate::configGroupName() const
{
    return u"KMailPluginEditorInit"_s;
}

QString PluginEditorInitManagerPrivate::configPrefixSettingKey() const
{
    return u"PluginEditorInit"_s;
}

QList<TextAddonsWidgets::PluginUtilData> PluginEditorInitManagerPrivate::pluginsDataList() const
{
    return mPluginDataList;
}

void PluginEditorInitManagerPrivate::initializePlugins()
{
    const QList<KPluginMetaData> plugins = KPluginMetaData::findPlugins(u"pim6/kmail/plugineditorinit"_s);

    const TextAddonsWidgets::PluginUtil::PluginsStateList pair = PimCommon::PluginUtil::loadPluginSetting(configGroupName(), configPrefixSettingKey());

    QListIterator<KPluginMetaData> i(plugins);
    i.toBack();
    while (i.hasPrevious()) {
        PluginEditorInitInfo info;
        const KPluginMetaData data = i.previous();

        // 1) get plugin data => name/description etc.
        info.pluginData = TextAddonsWidgets::PluginUtil::createPluginMetaData(data);
        // 2) look at if plugin is activated
        const bool isPluginActivated = TextAddonsWidgets::PluginUtil::isPluginActivated(pair.enabledPluginList,
                                                                                        pair.disabledPluginList,
                                                                                        info.pluginData.mEnableByDefault,
                                                                                        info.pluginData.mIdentifier);
        info.isEnabled = isPluginActivated;
        info.metaDataFileNameBaseName = QFileInfo(data.fileName()).baseName();
        info.metaDataFileName = data.fileName();
        info.data = data;
        if (pluginEditorInitVersion() == data.version()) {
            info.plugin = nullptr;
            mPluginList.push_back(info);
        } else {
            qCWarning(MESSAGECOMPOSER_LOG) << "Plugin " << data.name() << " doesn't have correction plugin version. It will not be loaded.";
        }
    }
    QList<PluginEditorInitInfo>::iterator end(mPluginList.end());
    for (QList<PluginEditorInitInfo>::iterator it = mPluginList.begin(); it != end; ++it) {
        loadPlugin(&(*it));
    }
}

void PluginEditorInitManagerPrivate::loadPlugin(PluginEditorInitInfo *item)
{
    if (auto plugin = KPluginFactory::instantiatePlugin<PluginEditorInit>(item->data, q, QVariantList() << item->metaDataFileName).plugin) {
        item->plugin = plugin;
        item->plugin->setIsEnabled(item->isEnabled);
        item->pluginData.mHasConfigureDialog = item->plugin->hasConfigureDialog();
        mPluginDataList.append(item->pluginData);
    }
}

QList<PluginEditorInit *> PluginEditorInitManagerPrivate::pluginsList() const
{
    QList<PluginEditorInit *> lst;
    QList<PluginEditorInitInfo>::ConstIterator end(mPluginList.constEnd());
    for (QList<PluginEditorInitInfo>::ConstIterator it = mPluginList.constBegin(); it != end; ++it) {
        if (auto plugin = (*it).plugin) {
            lst << plugin;
        }
    }
    return lst;
}

PluginEditorInit *PluginEditorInitManagerPrivate::pluginFromIdentifier(const QString &id)
{
    QList<PluginEditorInitInfo>::ConstIterator end(mPluginList.constEnd());
    for (QList<PluginEditorInitInfo>::ConstIterator it = mPluginList.constBegin(); it != end; ++it) {
        if ((*it).pluginData.mIdentifier == id) {
            return (*it).plugin;
        }
    }
    return {};
}

PluginEditorInitManager::PluginEditorInitManager(QObject *parent)
    : QObject(parent)
    , d(new MessageComposer::PluginEditorInitManagerPrivate(this))
{
}

PluginEditorInitManager::~PluginEditorInitManager() = default;

PluginEditorInitManager *PluginEditorInitManager::self()
{
    static PluginEditorInitManager s_self;
    return &s_self;
}

QList<PluginEditorInit *> PluginEditorInitManager::pluginsList() const
{
    return d->pluginsList();
}

QString PluginEditorInitManager::configGroupName() const
{
    return d->configGroupName();
}

QString PluginEditorInitManager::configPrefixSettingKey() const
{
    return d->configPrefixSettingKey();
}

QList<TextAddonsWidgets::PluginUtilData> PluginEditorInitManager::pluginsDataList() const
{
    return d->pluginsDataList();
}

PluginEditorInit *PluginEditorInitManager::pluginFromIdentifier(const QString &id)
{
    return d->pluginFromIdentifier(id);
}

#include "moc_plugineditorinitmanager.cpp"
