/*
  SPDX-FileCopyrightText: 2009 Klaralvdalens Datakonsult AB, a KDAB Group company, info@kdab.net
  SPDX-FileCopyrightText: 2009 Leo Franchi <lfranchi@kde.org>

  SPDX-License-Identifier: LGPL-2.0-or-later
*/

#include "job/signencryptjob.h"
using namespace Qt::Literals::StringLiterals;

#include "contentjobbase_p.h"
#include "job/protectedheadersjob.h"
#include "utils/util_p.h"

#include <QGpgME/Protocol>
#include <QGpgME/SignEncryptJob>

#include "messagecomposer_debug.h"
#include <KMime/Content>
#include <KMime/Headers>
#include <KMime/Message>
#include <Libkleo/Formatting>

#include <gpgme++/encryptionresult.h>
#include <gpgme++/global.h>
#include <gpgme++/signingresult.h>

using namespace MessageComposer;

class MessageComposer::SignEncryptJobPrivate : public ContentJobBasePrivate
{
public:
    explicit SignEncryptJobPrivate(SignEncryptJob *qq)
        : ContentJobBasePrivate(qq)
    {
    }

    std::vector<GpgME::Key> signers;

    std::vector<GpgME::Key> encKeys;
    QStringList recipients;
    Kleo::CryptoMessageFormat format;
    std::unique_ptr<KMime::Content> content;
    KMime::Message *skeletonMessage = nullptr;

    bool protectedHeaders = true;
    bool protectedHeadersObvoscate = false;

    // copied from messagecomposer.cpp
    [[nodiscard]] bool binaryHint(Kleo::CryptoMessageFormat f)
    {
        switch (f) {
        case Kleo::SMIMEFormat:
        case Kleo::SMIMEOpaqueFormat:
            return true;
        default:
        case Kleo::OpenPGPMIMEFormat:
        case Kleo::InlineOpenPGPFormat:
            return false;
        }
    }

    Q_DECLARE_PUBLIC(SignEncryptJob)
};

SignEncryptJob::SignEncryptJob(QObject *parent)
    : ContentJobBase(*new SignEncryptJobPrivate(this), parent)
{
}

SignEncryptJob::~SignEncryptJob() = default;

void SignEncryptJob::setContent(std::unique_ptr<KMime::Content> &&content)
{
    Q_D(SignEncryptJob);

    Q_ASSERT(content);

    d->content = std::move(content);
}

void SignEncryptJob::setCryptoMessageFormat(Kleo::CryptoMessageFormat format)
{
    Q_D(SignEncryptJob);

    // There *must* be a concrete format set at this point.
    Q_ASSERT(format == Kleo::OpenPGPMIMEFormat || format == Kleo::InlineOpenPGPFormat || format == Kleo::SMIMEFormat || format == Kleo::SMIMEOpaqueFormat);
    d->format = format;
}

void SignEncryptJob::setSigningKeys(const std::vector<GpgME::Key> &signers)
{
    Q_D(SignEncryptJob);

    d->signers = signers;
}

void SignEncryptJob::setEncryptionKeys(const std::vector<GpgME::Key> &keys)
{
    Q_D(SignEncryptJob);

    d->encKeys = keys;
}

void SignEncryptJob::setRecipients(const QStringList &recipients)
{
    Q_D(SignEncryptJob);

    d->recipients = recipients;
}

void SignEncryptJob::setSkeletonMessage(KMime::Message *skeletonMessage)
{
    Q_D(SignEncryptJob);

    d->skeletonMessage = skeletonMessage;
}

void SignEncryptJob::setProtectedHeaders(bool protectedHeaders)
{
    Q_D(SignEncryptJob);

    d->protectedHeaders = protectedHeaders;
}

void SignEncryptJob::setProtectedHeadersObvoscate(bool protectedHeadersObvoscate)
{
    Q_D(SignEncryptJob);

    d->protectedHeadersObvoscate = protectedHeadersObvoscate;
}

QStringList SignEncryptJob::recipients() const
{
    Q_D(const SignEncryptJob);

    return d->recipients;
}

std::vector<GpgME::Key> SignEncryptJob::encryptionKeys() const
{
    Q_D(const SignEncryptJob);

    return d->encKeys;
}

void SignEncryptJob::doStart()
{
    Q_D(SignEncryptJob);
    Q_ASSERT(d->resultContent == nullptr); // Not processed before.

    if (d->protectedHeaders && d->skeletonMessage && d->format & Kleo::OpenPGPMIMEFormat) {
        auto pJob = new ProtectedHeadersJob;
        pJob->setContent(std::move(d->content));
        pJob->setSkeletonMessage(d->skeletonMessage);
        pJob->setObvoscate(d->protectedHeadersObvoscate);
        appendSubjob(pJob);
    }

    ContentJobBase::doStart();
}

void SignEncryptJob::slotResult(KJob *job)
{
    Q_D(SignEncryptJob);
    if (error() || job->error()) {
        ContentJobBase::slotResult(job);
        return;
    }

    if (auto cjob = qobject_cast<ContentJobBase *>(job); cjob && !qobject_cast<ProtectedHeadersJob *>(job)) {
        // clone input content as ComposerJob uses the same job providing that
        // for potntially multiple encryption jobs (which end up modifying this)
        d->content = cjob->content()->clone();

        // forward input to the ProtectedHeadersJob if there's one, bypassing ContentJobBase's result propagation
        if (subjobs().size() == 2) {
            if (auto pjob = static_cast<ProtectedHeadersJob *>(subjobs().last()); pjob) {
                pjob->setContent(std::move(d->content));
            }
        }

        // skip ContentJobBase as we passed on the subjob result here and don't want to consume it ourselves'
        KCompositeJob::slotResult(job);
        d->doNextSubjob();
        return;
    }

    ContentJobBase::slotResult(job);
}

void SignEncryptJob::process()
{
    Q_D(SignEncryptJob);
    Q_ASSERT(d->resultContent == nullptr); // Not processed before.

    // if setContent hasn't been called, we assume that a subjob was added
    // and we want to use that
    if (!d->content || !d->content->hasContent()) {
        Q_ASSERT(d->subjobContents.size() == 1);
        d->content = std::move(d->subjobContents.front());
    }

    const QGpgME::Protocol *proto = nullptr;
    if (d->format & Kleo::AnyOpenPGP) {
        proto = QGpgME::openpgp();
    } else if (d->format & Kleo::AnySMIME) {
        proto = QGpgME::smime();
    } else {
        return;
    }
    Q_ASSERT(proto);

    qCDebug(MESSAGECOMPOSER_LOG) << "creating signencrypt from:" << proto->name() << proto->displayName();

    d->content->assemble();

    // replace simple LFs by CRLFs for all MIME supporting CryptPlugs
    // according to RfC 2633, 3.1.1 Canonicalization
    QByteArray content;
    if (d->format & Kleo::InlineOpenPGPFormat) {
        content = d->content->body();
    } else if (!(d->format & Kleo::SMIMEOpaqueFormat)) {
        content = KMime::LFtoCRLF(d->content->encodedContent());
    } else { // SMimeOpaque doesn't need LFtoCRLF, else it gets munged
        content = d->content->encodedContent();
    }

    QGpgME::SignEncryptJob *job(proto->signEncryptJob(!d->binaryHint(d->format), d->format == Kleo::InlineOpenPGPFormat));
    QObject::connect(job,
                     &QGpgME::SignEncryptJob::result,
                     this,
                     [this, d](const GpgME::SigningResult &signingResult,
                               const GpgME::EncryptionResult &encryptionResult,
                               const QByteArray &cipherText,
                               const QString &auditLogAsHtml,
                               const GpgME::Error &auditLogError) {
                         Q_UNUSED(auditLogAsHtml)
                         Q_UNUSED(auditLogError)
                         if (signingResult.error()) {
                             qCDebug(MESSAGECOMPOSER_LOG) << "signing failed:" << Kleo::Formatting::errorAsString(signingResult.error());
                             setError(signingResult.error().code());
                             setErrorText(Kleo::Formatting::errorAsString(signingResult.error()));
                             emitResult();
                             return;
                         }
                         if (encryptionResult.error()) {
                             qCDebug(MESSAGECOMPOSER_LOG) << "encrypting failed:" << Kleo::Formatting::errorAsString(encryptionResult.error());
                             setError(encryptionResult.error().code());
                             setErrorText(Kleo::Formatting::errorAsString(encryptionResult.error()));
                             emitResult();
                             return;
                         }

                         QByteArray signatureHashAlgo = signingResult.createdSignature(0).hashAlgorithmAsString();
                         d->resultContent =
                             MessageComposer::Util::composeHeadersAndBody(std::move(d->content), cipherText, d->format, false, signatureHashAlgo);

                         emitResult();
                     });

    const auto error = job->start(d->signers, d->encKeys, content, false);
    if (error.code()) {
        job->deleteLater();
        setError(error.code());
        setErrorText(Kleo::Formatting::errorAsString(error));
        emitResult();
    }
}

#include "moc_signencryptjob.cpp"
