// SPDX-FileCopyrightText: 2023 Carl Schwan <carlschwan@kde.org>
// SPDX-License-Identifier: LGPL-2.1-or-later

#include "../config-merkuro.h"
#include "contactimageprovider.h"
#include "messagehandler.h"
#include <KAboutData>
#include <KCrash>
#include <KDBusService>
#include <KLocalizedQmlContext>
#include <KLocalizedString>
#include <KWindowSystem>
#include <QApplication>
#include <QCommandLineParser>
#include <QDir>
#include <QIcon>
#include <QQmlApplicationEngine>
#include <QQmlContext>
#include <QQuickStyle>
#include <QQuickWindow>

#include <KIconTheme>

#include <KStyleManager>
using namespace Qt::Literals::StringLiterals;
static void raiseWindow(QWindow *window)
{
    KWindowSystem::updateStartupId(window);
    KWindowSystem::activateWindow(window);
}

int main(int argc, char *argv[])
{
    KIconTheme::initTheme();
    QGuiApplication::setAttribute(Qt::AA_ShareOpenGLContexts);
    QApplication app(argc, argv);
    KLocalizedString::setApplicationDomain("merkuro"_ba);
    QCoreApplication::setOrganizationName(u"KDE"_s);
    QCoreApplication::setApplicationName(u"Merkuro Mail"_s);
    QCoreApplication::setQuitLockEnabled(false);

    // Default to org.kde.desktop style unless the user forces another style
    if (qEnvironmentVariableIsEmpty("QT_QUICK_CONTROLS_STYLE")) {
        QQuickStyle::setStyle(u"org.kde.desktop"_s);
    }

    KStyleManager::initStyle();

    KAboutData aboutData(
        // The program name used internally.
        u"merkuro.mail"_s,
        // A displayable program name string.
        i18nc("@title", "Merkuro Mail"),
        QStringLiteral(MERKURO_VERSION_STRING),
        // Short description of what the app does.
        i18n("Email Client"),
        // The license this code is released under.
        KAboutLicense::GPL_V3,
        // Copyright Statement.
        i18n("© KDE Community 2021–2026"));
    aboutData.setBugAddress("https://bugs.kde.org/enter_bug.cgi?format=guided&product=merkuro&version=" + QStringLiteral(MERKURO_VERSION_STRING).toUtf8());
    aboutData.addAuthor(i18nc("@info:credit", "Carl Schwan"),
                        i18nc("@info:credit", "Maintainer"),
                        u"carl@carlschwan.eu"_s,
                        u"https://carlschwan.eu"_s,
                        QUrl(u"https://carlschwan.eu/avatar.png"_s));
    aboutData.addAuthor(i18nc("@info:credit", "Clau Cambra"),
                        i18nc("@info:credit", "Maintainer"),
                        u"claudio.cambra@gmail.com"_s,
                        u"https://claudiocambra.com"_s);
    KAboutData::setApplicationData(aboutData);
    KCrash::initialize();
    QGuiApplication::setWindowIcon(QIcon::fromTheme(u"org.kde.merkuro.mail"_s));

    QCommandLineParser parser;
    aboutData.setupCommandLine(&parser);
    parser.process(app);
    aboutData.processCommandLine(&parser);

    KDBusService service(KDBusService::Unique);

    const auto options = parser.optionNames();
    const auto args = parser.positionalArguments();
    QQmlApplicationEngine engine;

    engine.addImageProvider(u"contact"_s, new ContactImageProvider);

    engine.rootContext()->setContextObject(new KLocalizedQmlContext(&engine));
    if (!args.isEmpty()) {
        engine.loadFromModule("org.kde.merkuro.mail", "OpenMbox");
        const auto rootObjects = engine.rootObjects();
        if (rootObjects.isEmpty()) {
            return -1;
        }
        parser.process(app);

        const QStringList args = parser.positionalArguments();
        for (auto obj : rootObjects) {
            auto view = qobject_cast<QQuickWindow *>(obj);
            auto messageHandler = view->findChild<MessageHandler *>(u"MessageHandler"_s);
            const auto file = QUrl::fromUserInput(args.at(args.count() - 1), QDir::currentPath());
            messageHandler->open(file);
        }
    } else {
        engine.loadFromModule("org.kde.merkuro.mail", "Main");

        QObject::connect(&service,
                         &KDBusService::activateRequested,
                         &engine,
                         [&engine](const QStringList & /*arguments*/, const QString & /*workingDirectory*/) {
                             const auto rootObjects = engine.rootObjects();
                             for (auto obj : rootObjects) {
                                 auto view = qobject_cast<QQuickWindow *>(obj);
                                 if (view) {
                                     raiseWindow(view);
                                     return;
                                 }
                             }
                         });
    }

    if (engine.rootObjects().isEmpty()) {
        return -1;
    }

    return app.exec();
}
