/*
    SPDX-FileCopyrightText: 2021 Aditya Mehra <aix.m@outlook.com>
    SPDX-FileCopyrightText: 2019 Marco Martin <mart@kde.org>

    SPDX-License-Identifier: GPL-2.0-or-later
*/

#include "biglauncherhomescreen.h"
#include "applicationlistmodel.h"
#include "favslistmodel.h"
#include "shortcuts.h"

#include <QProcess>
#include <QQmlEngine>
#include <QQmlContext>
#include <sessionmanagement.h>

#include <QDBusConnection>
#include <QDBusMessage>

static QObject *favsManagerSingletonProvider(QQmlEngine *engine, QJSEngine *scriptEngine)
{
    Q_UNUSED(scriptEngine);

    //singleton managed internally, qml should never delete it
    engine->setObjectOwnership(FavsManager::instance(), QQmlEngine::CppOwnership);
    return FavsManager::instance();
}

static QObject *shortcutsSingletonProvider(QQmlEngine *engine, QJSEngine *scriptEngine)
{
    Q_UNUSED(scriptEngine);

    //singleton managed internally, qml should never delete it
    engine->setObjectOwnership(Shortcuts::instance(), QQmlEngine::CppOwnership);
    return Shortcuts::instance();
}

static QObject *dbusSingletonProvider(QQmlEngine *engine, QJSEngine *scriptEngine)
{
    Q_UNUSED(scriptEngine);

    engine->setObjectOwnership(BigLauncherDbusAdapterInterface::instance(), QQmlEngine::CppOwnership);
    return BigLauncherDbusAdapterInterface::instance();
}

HomeScreen::HomeScreen(QObject *parent, const KPluginMetaData &data, const QVariantList &args)
    : Plasma::Containment(parent, data, args)
    , m_applicationListModel(new ApplicationListSearchModel(this, new ApplicationListModel(this)))
    , m_session(new SessionManagement(this))
    , m_favsListModel(new FavsListModel(FavsManager::instance(), this))
{
    // Initialize DBus adapter with configuration
    BigLauncherDbusAdapterInterface::instance()->init(config().group(QStringLiteral("General")));

    const char *uri = "org.kde.private.biglauncher";
    qmlRegisterSingletonType<Shortcuts>(uri, 1, 0, "Shortcuts", shortcutsSingletonProvider);
    qmlRegisterSingletonType<FavsManager>(uri, 1, 0, "FavsManager", favsManagerSingletonProvider);
    qmlRegisterSingletonType<BigLauncherDbusAdapterInterface>(uri, 1, 0, "BigLauncherDbusAdapterInterface", dbusSingletonProvider);
    qmlRegisterUncreatableType<ApplicationListModel>(uri, 1, 0, "ApplicationListModel", QStringLiteral("Cannot create an item of type ApplicationListModel"));
    qmlRegisterUncreatableType<ApplicationListSearchModel>(uri,
                                                           1,
                                                           0,
                                                           "ApplicationListSearchModel",
                                                           QStringLiteral("Cannot create an item of type ApplicationListSearchModel"));
    qmlRegisterUncreatableType<FavsListModel>(uri, 1, 0, "FavsListModel", QStringLiteral("Cannot create an item of type FavsListModel"));

    setHasConfigurationInterface(true);
}

HomeScreen::~HomeScreen()
{
}

ApplicationListSearchModel *HomeScreen::applicationListModel() const
{
    return m_applicationListModel;
}

FavsListModel *HomeScreen::favsListModel() const
{
    return m_favsListModel;
}

void HomeScreen::openSettings(QString module)
{
    Q_EMIT minimizeAllTasksRequested();

    if (module.isEmpty()) {
        executeCommand(QStringLiteral("plasma-bigscreen-settings"));
    } else {
        executeCommand(QStringLiteral("plasma-bigscreen-settings -m ") + module);
    }
}

void HomeScreen::openSearch()
{
    Q_EMIT openSearchRequested();
}

void HomeScreen::openTasks()
{
    Q_EMIT openTasksRequested();
}

void HomeScreen::openHomeOverlay()
{
    Q_EMIT openHomeOverlayRequested();
}

void HomeScreen::showOSD(const QString &text, const QString &iconName)
{
    QDBusMessage msg = QDBusMessage::createMethodCall(QStringLiteral("org.kde.plasmashell"),
                                                      QStringLiteral("/org/kde/osdService"),
                                                      QStringLiteral("org.kde.osdService"),
                                                      QStringLiteral("showText"));

    msg.setArguments({iconName, text});

    QDBusConnection::sessionBus().call(msg, QDBus::NoBlock);
}

void HomeScreen::showAppLaunchScreen(const QString &appName, const QString &icon)
{
    Q_EMIT showAppLaunchScreenRequested(appName, icon);
}

void HomeScreen::executeCommand(const QString &command)
{
    qInfo() << "Executing" << command;
    QStringList split = QProcess::splitCommand(command);
    QProcess::startDetached(split.takeFirst(), split);
}

K_PLUGIN_CLASS_WITH_JSON(HomeScreen, "metadata.json")

#include "biglauncherhomescreen.moc"
#include "moc_biglauncherhomescreen.cpp"
