// SPDX-FileCopyrightText: 2023 Devin Lin <devin@kde.org>
// SPDX-FileCopyrightText: 2025 Kristen McWilliam <kristen@kde.org>
//
// SPDX-License-Identifier: GPL-2.0-or-later

pragma ComponentBehavior: Bound

import QtQuick
import QtQuick.Controls
import QtQuick.Layouts

import org.kde.kirigami as Kirigami
import org.kde.plasmasetup
import org.kde.plasmasetup.components as PlasmaSetupComponents

Kirigami.Page {
    id: root

    leftPadding: 0
    rightPadding: 0
    topPadding: 0
    bottomPadding: 0

    property int currentIndex: -1
    readonly property int stepCount: stepsRepeater.count
    property bool showingLanding: true

    // filled by items
    property Control currentStepItem: null
    property Control nextStepItem: null
    property Control previousStepItem: null
    property PlasmaSetupComponents.SetupModule currentModule: null

    readonly property bool onFinalPage: currentIndex === (stepCount - 1)

    // step animation
    // manually doing the animation is more performant and less glitchy with window resize than a SwipeView
    property real previousStepItemX: 0
    property real currentStepItemX: 0
    property real nextStepItemX: 0

    NumberAnimation on previousStepItemX {
        id: previousStepAnim
        duration: 400
        easing.type: Easing.OutExpo
        onFinished: {
            if (root.previousStepItemX != 0) {
                root.previousStepItem.visible = false;
            }
        }
    }

    NumberAnimation on currentStepItemX {
        id: currentStepAnim
        duration: 400
        easing.type: Easing.OutExpo
    }

    NumberAnimation on nextStepItemX {
        id: nextStepAnim
        duration: 400
        easing.type: Easing.OutExpo
        onFinished: {
            if (root.nextStepItemX != 0) {
                root.nextStepItem.visible = false;
            }
        }
    }

    onStepCountChanged: {
        // reset position
        requestPreviousPage();
    }

    function finishFinalPage(): void {
        // Finalize the initial setup process and exit the wizard.
        InitialStartUtil.finish();
    }

    function requestNextPage(): void {
        if (previousStepAnim.running || currentStepAnim.running || nextStepAnim.running) {
            return;
        }

        previousStepItemX = 0;

        // Notify the next page/module it is being activated.
        //
        // Requires the module to implement an `onPageActivated` function.
        //
        // This allows the module to perform any necessary setup, and check if
        // any data it relies on has been updated since the last activation.
        if (currentIndex + 1 < stepCount) {
            let nextItem = stepsRepeater.itemAt(currentIndex + 1);
            if (nextItem && nextItem.module && typeof nextItem.module.onPageActivated === "function") {
                nextItem.module.onPageActivated();
            }
        }

        currentIndex++;
        stepHeading.changeText(currentStepItem.name);

        currentStepItemX = root.width;
        currentStepItem.visible = true;

        previousStepAnim.to = -root.width;
        previousStepAnim.restart();
        currentStepAnim.to = 0;
        currentStepAnim.restart();
    }

    function requestPreviousPage(): void {
        if (previousStepAnim.running || currentStepAnim.running || nextStepAnim.running) {
            return;
        }

        if (currentIndex === 0) {
            root.showingLanding = true;
            landingComponent.returnToLanding();
        } else {
            nextStepItemX = 0;

            currentIndex--;
            stepHeading.changeText(currentStepItem.name);

            currentStepItemX = -root.width;
            currentStepItem.visible = true;

            nextStepAnim.to = root.width;
            nextStepAnim.restart();
            currentStepAnim.to = 0;
            currentStepAnim.restart();
        }
    }

    LandingComponent {
        id: landingComponent
        anchors.fill: parent

        onRequestNextPage: {
            root.showingLanding = false;
            stepHeading.changeText(root.currentStepItem.name);
        }
    }

    PagesModel {
        id: pagesModel

        Component.onCompleted: reload()

        onLoaded: root.currentIndex = 0
    }

    Connections {
        target: pagesModel

        // onDataChanged will be emitted if the model reloads the translations
        // after the user chooses a language. We need to catch that since we are
        // using `changeText()` instead of a property binding.
        function onDataChanged(): void {
            if (root.currentIndex >= 0 && !root.showingLanding) {
                // Update the heading with the fresh translation from the model
                stepHeading.changeText(root.currentStepItem.name);
            }
        }
    }

    Item {
        id: stepsComponent
        anchors.fill: parent

        // animation when we switch to step stage
        opacity: root.showingLanding ? 0 : 1
        property real translateY: root.showingLanding ? overlaySteps.height : 0

        Behavior on opacity {
            NumberAnimation {
                duration: 1000
                easing.type: Easing.OutExpo
            }
        }

        Behavior on translateY {
            NumberAnimation {
                duration: 1000
                easing.type: Easing.OutExpo
            }
        }

        transform: Translate {
            y: stepsComponent.translateY
        }

        Rectangle {
            id: overlaySteps

            Kirigami.Theme.inherit: false
            Kirigami.Theme.colorSet: Kirigami.Theme.Window

            color: Kirigami.Theme.backgroundColor
            clip: true

            radius: Kirigami.Settings.isMobile ? 0 : Kirigami.Units.cornerRadius + 8

            anchors {
                fill: Kirigami.Settings.isMobile ? parent : undefined
                topMargin: Kirigami.Settings.isMobile ? root.height * 0.3 : undefined
                centerIn: Kirigami.Settings.isMobile ? undefined : parent
            }

            width: Kirigami.Settings.isMobile ? undefined : parent.width * 0.4
            height: Kirigami.Settings.isMobile ? undefined : parent.height * 0.7

            Behavior on height {
                NumberAnimation {
                    duration: Kirigami.Units.longDuration
                    easing.type: Easing.InOutQuad
                }
            }

            // all steps are in this container
            ColumnLayout {
                id: container

                anchors.fill: parent

                // heading for all the wizard steps
                Label {
                    id: stepHeading
                    opacity: 0
                    horizontalAlignment: Text.AlignHCenter
                    font.pointSize: 18

                    Layout.fillWidth: true
                    Layout.alignment: Qt.AlignHCenter
                    Layout.topMargin: Kirigami.Units.gridUnit

                    property string toText

                    function changeText(text: string): void {
                        toText = text;
                        toHidden.restart();
                    }

                    NumberAnimation on opacity {
                        id: toHidden
                        duration: 200
                        to: 0
                        onFinished: {
                            stepHeading.text = stepHeading.toText;
                            toShown.restart();
                        }
                    }

                    NumberAnimation on opacity {
                        id: toShown
                        duration: 200
                        to: 1
                    }
                }

                Item {
                    Layout.fillWidth: true
                    Layout.fillHeight: true

                    // setup steps
                    Repeater {
                        id: stepsRepeater
                        model: pagesModel
                        delegate: PageDelegate {}
                        Layout.alignment: Qt.AlignHCenter
                    }
                }

                // bottom footer
                RowLayout {
                    id: stepFooter

                    Layout.fillWidth: true
                    Layout.alignment: Qt.AlignBottom
                    Layout.margins: Kirigami.Units.gridUnit

                    Button {
                        Layout.alignment: Qt.AlignLeft

                        text: i18nc("@action:button", "Back")
                        icon.name: "arrow-left-symbolic"

                        onClicked: root.requestPreviousPage()
                    }

                    Item {
                        Layout.fillWidth: true
                    }

                    Button {
                        Layout.alignment: Qt.AlignRight
                        // Nicer to have the arrow on the side it's pointing to
                        LayoutMirroring.enabled: Qt.application.layoutDirection === Qt.LeftToRight // qmllint disable missing-property

                        visible: !root.onFinalPage
                        text: i18nc("@action:button", "Next")
                        icon.name: "arrow-right-symbolic"

                        enabled: root.currentModule.nextEnabled

                        onClicked: root.requestNextPage()
                    }

                    Button {
                        Layout.alignment: Qt.AlignRight

                        visible: root.onFinalPage
                        text: i18nc("@action:button", "Finish")
                        icon.name: "dialog-ok-symbolic"

                        enabled: root.currentModule.nextEnabled

                        onClicked: {
                            // Ensure the `Finish` button can only be click once.
                            root.currentModule.nextEnabled = false;
                            // Finalize and exit the wizard.
                            root.finishFinalPage();
                        }
                    }
                }
            }
        }
    }

    /*!
    * Delegate that represents each page in the wizard.
    */
    component PageDelegate: Control {
        id: item

        required property int index
        required property string name

        property PlasmaSetupComponents.SetupModule module: null

        Component.onCompleted: {
            module = pagesModel.pageItem(index);
            updateRootItems();
        }

        visible: index === 0 // the binding is broken later
        contentItem: module?.contentItem

        Binding {
            target: item.module
            property: 'cardWidth'
            value: Math.min(Kirigami.Units.gridUnit * 30, item.contentItem.width - Kirigami.Units.gridUnit * 2)
        }

        clip: true
        topPadding: Kirigami.Units.gridUnit
        bottomPadding: 0
        leftPadding: Kirigami.Units.gridUnit
        rightPadding: Kirigami.Units.gridUnit

        transform: Translate {
            x: {
                if (item.index === root.currentIndex - 1) {
                    return root.previousStepItemX;
                } else if (item.index === root.currentIndex + 1) {
                    return root.nextStepItemX;
                } else if (item.index === root.currentIndex) {
                    return root.currentStepItemX;
                }
                return 0;
            }
        }

        width: parent.width
        height: parent.height

        function updateRootItems(): void {
            if (index === root.currentIndex) {
                root.currentStepItem = item;
                root.currentModule = module;
            } else if (index === root.currentIndex - 1) {
                root.previousStepItem = item;
            } else if (index === root.currentIndex + 1) {
                root.nextStepItem = item;
            }
        }

        // keep root properties updated
        Connections {
            target: root

            function onCurrentIndexChanged(): void {
                item.updateRootItems();
            }
        }
    }
}
